import json
import logging
import datetime
import time
import requests
from sandbox.projects.media.resources import MediaJugglerCalls
from sandbox import sdk2


class MediaJugglerReport(sdk2.Task):
    """Send juggler reports to telegram and solomon"""

    class Parameters(sdk2.Parameters):
        service_name = sdk2.parameters.String('Service name', required=True)
        time_in_hour = sdk2.parameters.Integer('Period in hours', default='24')

        with sdk2.parameters.Group("Telegram settings") as telegram_settings:
            telegram = sdk2.parameters.Bool('Send stat to telegarm')
            chat_id = sdk2.parameters.String('Telegram chat id')

        with sdk2.parameters.Group("Juggler settings") as juggler_settings:
            login = sdk2.parameters.String('Login', required=True)
            method = sdk2.parameters.String('Method', required=True)

        with sdk2.parameters.Group("Solomon settings") as solomon_settings:
            solomon = sdk2.parameters.Bool('Enable solomon statistics')
            solomon_project = sdk2.parameters.String('Solomon project', required=True)
            solomon_cluster = sdk2.parameters.String('Solomon cluster', required=True)
            solomon_service = sdk2.parameters.String('Solomon service', required=True)
            solomon_sensor_name = sdk2.parameters.String('Solomon sensor name',
                                                         default='phone_escalation_cnt')

            secrets = sdk2.parameters.YavSecret("Yav secret with key: solomon_oauth_token",
                                                required=True)

    def send_to_tg(self, chat_id, text):
        if self.Parameters.telegram:
            headers = {"Accept": "application/json", "Content-Type": "application/json"}
            cert = "/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt"
            payload = json.dumps({"text": text})
            url = "https://api2.nbt.media.yandex.net"
            uri = "%s/v2/proxy/telegram/raw/%s" % (url, chat_id)
            logging.info(payload)
            try:
                req = requests.post(uri, headers=headers, data=payload, verify=cert)
                logging.info(req.text)
                req.raise_for_status()
                return True
            except requests.exceptions.HTTPError as err:
                logging.info(err)
                return False
        else:
            return True

    @staticmethod
    def checks(login, method, since, until):
        headers = {"Accept": "application/json", "Content-Type": "application/json"}
        cert = "/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt"
        payload = '{ "filters":' \
                  '[{ "login": "%s", "method": "%s" }], ' \
                  '"until": %s, "since": %s, "resolution": 0}' % (login, method, until, since)

        url = 'https://juggler-api.search.yandex.net'
        uri = '%s/v2/history/get_notification_rollups' % url
        logging.info(payload)

        try:
            req = requests.post(uri, headers=headers, data=payload, verify=cert)
            logging.info(req.text)
            req.raise_for_status()
            return req
        except requests.exceptions.HTTPError as err:
            logging.info(err)
            return err

    @staticmethod
    def result_to_file(message, file_name="calls.txt"):
        with open(file_name, "a+") as file_handler:
            file_handler.write(message + '\n')

    def data(self):
        send = self.send_to_tg
        full_checks = self.checks

        now = datetime.datetime.now()
        since = now - datetime.timedelta(hours=self.Parameters.time_in_hour)
        until = now

        result = full_checks(self.Parameters.login,
                             self.Parameters.method,
                             since.strftime("%s"),
                             until.strftime("%s"))
        checks = result.json()

        final_list = []
        for check in checks['items']:
            ts = check['last_notification']['timestamp']
            datetime_ts = datetime.datetime.fromtimestamp(ts)
            if since < datetime_ts < until:
                final_list.append(check)

        if final_list:
            msg = 'there were %s calls from %s to %s' % (len(final_list),
                                                         since,
                                                         until)

            self.result_to_file(msg)
            send(self.Parameters.chat_id, msg)
            time.sleep(1)
        else:
            msg = "there were no calls from %s to %s" % (since, until)

            self.result_to_file(msg)
            send(self.Parameters.chat_id, msg)

            return False

        return final_list

    def solomon(self, checks):
        solomon_oauth = self.Parameters.secrets.data()["solomon_oauth_token"]

        headers = {
            "Accept": "application/json",
            "Content-Type": "application/json",
            "Authorization": "OAuth %s" % solomon_oauth,
            }
        url = "http://solomon.yandex.net"
        uri = "%s/api/v2/push?project=%s&cluster=%s&service=%s" % (url,
                                                                   self.Parameters.solomon_project,
                                                                   self.Parameters.solomon_cluster,
                                                                   self.Parameters.solomon_service)

        sensor = self.Parameters.solomon_sensor_name
        element = []

        for check in checks:
            ts = check['last_notification']['timestamp']
            utc_datetime = datetime.datetime.fromtimestamp(float(ts)) + \
                           datetime.timedelta(hours=3)
            utc_ts = utc_datetime.strftime("%s")
            labels = {'labels': {'sensor': sensor}, 'ts': int(utc_ts), 'value': 1}
            element.append(labels)

        payload = {}
        payload['sensors'] = element

        logging.info(payload)
        try:
            req = requests.post(uri, headers=headers, json=payload)
            logging.info(req.text)
            req.raise_for_status()
            return True
        except requests.exceptions.HTTPError as err:
            logging.info(err)
            return False

    def app(self, checks, method='resource'):
        send = self.send_to_tg
        for check in checks:
            notification = check['last_notification']
            service = notification['service_name']
            host = notification['host_name']
            description = notification['description']
            date = str(datetime.datetime.fromtimestamp(notification['timestamp']))
            url = 'https://juggler.yandex-team.ru'
            uri = 'check_details/?host=%s&service=%s&last=1DAY' % (host, service)
            juggler_check = '%s/%s' % (url, uri)
            msg_rollup = ''
            msg = 'date: %s\n\n' \
                  'main_notification:\n' \
                  '    host: %s\n' \
                  '    sevice: %s\n' \
                  '    description: %s\n' \
                  '    juggler: %s\n\n' % (date,
                                           host,
                                           service,
                                           description,
                                           juggler_check)
            if 'rollup' in check:
                msg_rollup = 'rollups_notification:\n'
                for rollup in check['rollup']['counts']:
                    msg_rollup += '    host: %s\n' \
                                  '    service: %s\n\n' % (rollup['host'],
                                                           rollup['service'])
            msg = msg + msg_rollup
            if method == 'telegram':
                send(self.Parameters.chat_id, msg)
                time.sleep(1)
            self.result_to_file(msg)
        resource = MediaJugglerCalls(self, "night_calls", "calls.txt")
        send(self.Parameters.chat_id, resource.http_proxy)
        time.sleep(1)
        send(self.Parameters.chat_id, "Finish")

    def on_execute(self):
        send = self.send_to_tg
        checks = self.data()
        if checks:
            if self.Parameters.solomon:
                self.solomon(checks)
            if len(checks) > 5:
                msg = 'Too many calls. See link below'
                send(self.Parameters.chat_id, msg)
                self.app(checks)
            else:
                self.app(checks, 'telegram')
