"""Task for sandbox about l7 weight monitoring"""
import json
import logging
import time
import requests
from sandbox import sdk2
from sandbox.sandboxsdk import environments


class MediaMonitoringBalancerWeight(sdk2.Task):
    """Monitoring weight on l7 balancer"""

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('jsonschema')
        ]

    class Parameters(sdk2.Parameters):
        """Input parameters for tasks"""
        service_name = sdk2.parameters.String('Service name', required=True)
        juggler_tags = sdk2.parameters.List("Tags for juggler")
        task_tags = sdk2.parameters.List("Tags for task")

        tokens = sdk2.parameters.YavSecret("Yav secret with nanny oauth token", required=True)

        json_string = sdk2.parameters.JSON('Json config as string', required=True)

    def juggler_raw(self, host, service, status, msg):
        """Send raw event to juggler"""
        default_tags = ["l7", "weight"]
        default_tags.append(host)
        if self.Parameters.task_tags:
            tags = set(self.Parameters.juggler_tags).union(set(default_tags))
            tags = json.dumps(list(tags))

        headers = {"Accept": "application/json", "Content-Type": "application/json"}
        payload = '''
                  {
                    "source": "sandbox.%s",
                    "events": [{
                                 "host": "%s",
                                 "service": "%s-weight",
                                 "status": "%s",
                                 "description": "%s",
                                 "tags": %s
                              }]
                  }
                  ''' % (self.__class__.__name__,
                         host,
                         service,
                         status,
                         msg,
                         tags)

        url = 'http://juggler-push.search.yandex.net'
        uri = '%s/events' % url

        logging.info(payload)
        logging.info(uri)

        try:
            req = requests.post(uri, headers=headers, data=payload)
            logging.info(req.text)
            req.raise_for_status()
            return req
        except requests.exceptions.HTTPError as err:
            return err

    def check_l7(self, validate=None):
        """Check l7 weight"""
        config = self.Parameters.json_string
        token = self.Parameters.tokens.data()

        config_schema = {
            "type" : "object",
            "properties" : {
                "max_closed_time" : {"type" : "number"},
                "balancer_id" : {"type" : "string"},
                "section_id" : {"type" : "string"}
            },
            "required": ["max_closed_time", "balancer_id", "section_id"]
        }

        nanny_url = "https://nanny.yandex-team.ru"

        for conf in config.values():
            if validate:
                validate(instance=conf, schema=config_schema)

            headers = {
                "Authorization": "OAuth %s" % (token["nanny_oauth"]),
            }
            its_url = 'https://ext.its.yandex-team.ru'
            its_uri = 'v2/l7/heavy/%s/weights/snapshots/?limit=1&skip=0' % conf["balancer_id"]
            cert = '/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt'

            request = requests.get('%s/%s' % (its_url, its_uri), headers=headers, verify=cert)
            data = request.json()

            weight_timestamp = data['result'][0]['change_info']['ctime'] / 1000
            locations = data['result'][0]['content']['sections'][conf["section_id"]]['locations']
            cur_time = time.time()

            for data_center in locations.items():
                dc_name = data_center[0]
                current_weight = data_center[1]['weight']
                default_weight = data_center[1]['default_weight']
                if (current_weight != default_weight) and \
                        (cur_time - weight_timestamp >= conf["max_closed_time"]):
                    msg = "2;current value %s " \
                          "differs from the default value %s too long (since %s), " \
                          "dc %s, balancer %s, section %s; " \
                          "%s/ui/#/l7heavy/%s/" % \
                        (current_weight,
                         default_weight,
                         time.ctime(weight_timestamp),
                         dc_name,
                         conf["balancer_id"],
                         conf["section_id"],
                         nanny_url,
                         conf["balancer_id"])
                    self.juggler_raw(conf["balancer_id"],
                                     conf["section_id"],
                                     'CRIT',
                                     msg)
                else:
                    self.juggler_raw(conf["balancer_id"],
                                     conf["section_id"],
                                     'OK',
                                     'OK')

    def add_tags(self):
        current_tags = set(self.Parameters.tags)
        tags = [char.upper() for char in self.Parameters.task_tags]
        self.Parameters.tags = list(set(current_tags).union(set(tags)))

    def on_execute(self):
        import backports
        reload(backports)
        import jsonschema

        self.check_l7(jsonschema.validate)
        if self.Parameters.task_tags:
            self.add_tags()
