import re
from os import path
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.resource_types import MONGODB_UTILS
import logging
from constants import YAV_OAUTH_TOKEN_ALIAS, DEFAULT_OWNER
from dumprestore import MongoWorker
from datetime import date


class MongoDumpRestore(sdk2.Task):
    """
    Dump/Restore mongodb to/from S3
    This task does make mongodump gzipped archive and upload it into S3
    """

    class Requirements(sdk2.Task.Requirements):
        """
        binary and others requirements
        """
        client_tags = sdk2.task.ctc.Tag.GENERIC & sdk2.task.ctc.Tag.LINUX_XENIAL
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('boto3'),
        ]

    class Parameters(sdk2.Task.Parameters):
        """
        Form parameters
        """
        with sdk2.parameters.Group('S3 storage settings') as s3_group:
            s3_bucket = sdk2.parameters.String("S3 bucket name", required=True)
            s3_key_prefix = sdk2.parameters.String("S3 key prefix", required=True,
                                                   description="Part of path after bucket name")
            s3_testing = sdk2.parameters.Bool("Use testing S3 installation", default=False)
        with sdk2.parameters.Group('Mongo backup settings') as mongo_group:
            m_url = sdk2.parameters.String("Comma-separated YC mongo host:port", required=True)
            m_extra_args = sdk2.parameters.String("Extra args for mongo dump/restore", default='')
            m_dump_db = sdk2.parameters.String("DB to dump", default='test')
            m_action = sdk2.parameters.String("Action to do: mongodump/mongorestore",
                                              default='mongodump')
            fake_run = sdk2.parameters.Bool("Do not really run, only show messages", default=False)
            mongodb_utils_resource_id = sdk2.parameters.Resource('Resourse with mongodb utils',
                                                                 description='Resource of type MONGODB_UTILS with correct version of utils',
                                                                 resource_type=MONGODB_UTILS,
                                                                 required=True)
        with sdk2.parameters.Group('Secret settings') as secrets_group:
            yav_token_alias = sdk2.parameters.String("YAV Oauth token Vault alias", required=False,
                                                     default='')
            yav_user = sdk2.parameters.String("User with access to yav secrets", required=False,
                                              default='')
            yav_mongo_secret = sdk2.parameters.String("YAV secret with mongo backup user",
                                                      required=False, default='',
                                                      description="Secret with key 'backup' and "
                                                                  "password in value")
            yav_s3_secret = sdk2.parameters.String("YAV secret with s3 access keys 'access_key' and"
                                                   "'secret_key'", required=False, default='')

    def get_sb_secret(self, secret_alias, owner=DEFAULT_OWNER):
        """
        Retrieves secret from Sandbox vault
        :param secret_alias: alias of the secret to retrieve
        :param owner: owner of the secret
        :return: secret data
        """
        logging.info("Get Sandbox Vault secret alias %s", secret_alias)
        return sdk2.Vault.data(owner, secret_alias)

    def get_yav_secret(self, secret_id, oauth_token=None):
        """
        Retrieves secret from YAV via HTTP
        :param oauth_token: oauth token to YAV for HTTP requests
        :param secret_id: id of the secret to retrieve
        :return:
        """
        logging.info("Get YAV secret %s", secret_id)

        import requests
        from base64 import b64decode

        api_url = 'https://vault-api.passport.yandex.net/1/versions/{}'.format(secret_id)
        secret = requests.get(api_url, headers={'Authorization': 'OAuth {}'.format(oauth_token)},
                              verify=False).json()
        if secret['status'] != 'ok':
            raise Exception('Error on getting YAV secret data: {0}'.format(secret['message']))

        logging.info("Got secrets from %s", secret_id)
        secret_data = {el['key']: b64decode(el['value']) for el in secret['version']['value']}
        return secret_data

    def parse_s3cfg(self, data):
        """
        pase .s3cfg for access and secret keys for s3
        :param data:
        :return: (access_key, secret_key)
        """
        data = re.findall(r'([\w_]+key)\s*=\s*([\S]+)', data)
        secret = {key.strip(): val.strip() for key, val in data}
        return secret

    def init_globals(self):
        """
        init main values(secrets)
        """
        if self.Parameters.yav_token_alias:
            logging.info("Defined yav token vault alias as %s", self.Parameters.yav_token_alias)
            self.yav_token = self.get_sb_secret(self.Parameters.yav_token_alias, self.owner)
            logging.info("Use yav token from parameter %s", self.Parameters.yav_token_alias)
        else:
            logging.info("Try to get yav token vault alias %s", YAV_OAUTH_TOKEN_ALIAS)
            self.yav_token = self.get_sb_secret(YAV_OAUTH_TOKEN_ALIAS, self.owner)

        if not self.yav_token:
            logging.error("Can't get YAV token. Unable to process without secrets access.")
            return

        self.s3sec = self.parse_s3cfg(self.get_yav_secret(self.Parameters.yav_s3_secret,
                                                          oauth_token=self.yav_token)["s3cmd"])
        self.s3_url = "https://s3.mds.yandex.net"
        if not self.s3sec:
            logging.debug("Failed to get s3 secret.")
            return

        self.mongo_secrets = self.get_yav_secret(self.Parameters.yav_mongo_secret,
                                                 oauth_token=self.yav_token)
        # here mongodb-linux-x86_64-ubuntu1604-4.0.0 - directory from MONGODB_UTILS archive

        mongodb_utils = sdk2.Resource[self.Parameters.mongodb_utils_resource_id]
        fs_path_mongodb_utils = str(sdk2.ResourceData(mongodb_utils).path)
        run_process(['tar', 'xfv', fs_path_mongodb_utils, '-C', '/tmp'])
        self.Context.path_to_bin = path.join('/tmp', 'mongodb-linux-*', 'bin/')
        self.Context.save()

        self.bindir = self.Context.path_to_bin
        logging.info("BIN: %s", self.bindir)

        logging.info("Secrets init done.")

    def build_config(self):
        """
        generate task config from input parameters
        :return: None
        """
        self.s3_config = {'s3_url': self.s3_url,
                          's3_bucket': self.Parameters.s3_bucket,
                          's3_key_prefix': '{}/{}'.format(self.Parameters.s3_key_prefix,
                                                          date.today().strftime("%Y%m%d"))
                          }
        self.s3_config.update(self.s3sec)
        self.action_config = {'db': self.Parameters.m_dump_db,
                              'user': 'backup',
                              'password': self.mongo_secrets['backup'],
                              'extra_args': self.Parameters.m_extra_args,
                              'url': self.Parameters.m_url,
                              'binary': self.bindir,
                              'action': self.Parameters.m_action}
        logging.info("Config build done.")

    def on_execute(self):
        self.init_globals()
        self.build_config()
        worker = MongoWorker(self.action_config, self.s3_config)
        worker.exec_action()
