import subprocess
import logging
import os
import re


class S3Client:
    """
    s3 download/upload
    """
    def __init__(self, config):
        self.log = logging.getLogger(self.__class__.__name__)
        self.prefix = '/tmp/dump_restore'
        self.bucket = config['s3_bucket']
        self.config = config
        self.s3cmd = self.s3_connect()

    def s3_connect(self):
        import boto3
        from boto3.s3.transfer import S3Transfer
        from botocore.exceptions import ClientError

        s3_endpoint_url = self.config['s3_url']
        s3_secret_key_id = self.config['access_key']
        s3_secret_key = self.config['secret_key']
        s3_bucket = self.config['s3_bucket']

        s3conn = boto3.client('s3', endpoint_url=s3_endpoint_url,
                              aws_access_key_id=s3_secret_key_id,
                              aws_secret_access_key=s3_secret_key)
        try:
            s3conn.head_bucket(Bucket=s3_bucket)
        except ClientError as err:
            error_code = int(err.response['Error']['Code'])
            if error_code == 404:
                s3conn.create_bucket(Bucket=s3_bucket)
            else:
                msg = 's3 get bucket code {}'.format(error_code)
                self.log.info("message: %s, resp: %s", msg, err.response['Error']['Message'])
                raise Exception(msg)
        self.log.info("S3 connected")
        return S3Transfer(s3conn)

    def push(self, filename, s3_path):
        from botocore.exceptions import ClientError

        self.log.info("Upload file %s...", filename)
        try:
            self.s3cmd.upload_file(filename, self.bucket, s3_path)
        except ClientError as err:
            self.log.error(err.response['Error'])
            raise

        self.log.info('Upload finished.')

    def pull(self, s3_path, filename):
        from botocore.exceptions import ClientError

        self.log.info("Download file: %s", filename)
        try:
            self.s3cmd.download_file(self.bucket, s3_path, filename)
        except ClientError as err:
            self.log.error(err.response['Error'])
            raise
        self.log.info('Download finished.')


class MongoWorker:
    """
    Mongo Dump and Restore class
    """
    def __init__(self, config, s3config):
        self.log = logging.getLogger(self.__class__.__name__)
        self.config = config
        self.prefix = '/tmp/dump_restore'
        self.log.debug("Run %s extra argrs: %s, mongo url: %s.",
                       config['action'], config['extra_args'], config['url'])
        self.s3_key_prefix = s3config['s3_key_prefix']
        self.s3client = S3Client(s3config)

    def gen_command(self, db):
        auth_args = ''
        opts = self.config['extra_args'] or ''
        passwd = ''
        target_db = db

        if self.config['user']:
            uname = self.config['user']
            passwd = self.config['password']
            auth_db = self.config['auth_db'] if 'auth_db' in self.config else target_db
            auth_args = '--authenticationDatabase {} -u {}'.format(auth_db, uname)

        util = self.config['binary'] + self.config['action']
        if self.config['action'] == 'mongorestore':
            fname = "{}/{}.gz".format(self.prefix, target_db)
            cmd = util + ' --gzip --drop --archive={}'.format(fname)
        elif self.config['action'] == 'mongodump':
            fname = "{}/{}.gz".format(self.prefix, target_db)
            cmd = util + ' --gzip --archive={} -d {}'.format(fname, target_db)
        else:
            self.log.debug('Unknown action %s', self.config['action'])
            return None

        if self.config['url']:
            cmd += " {} --host '{}'".format(opts, self.config['url'])
        else:
            self.log.error("Incorrect or not found mongodb url. Exit.")
            return None

        if auth_args:
            cmd += " {} ".format(auth_args)

        self.log.debug('Generated command to run: %s', cmd)
        # now after log add password
        if 'password' in self.config:
            cmd += ' -p {}'.format(passwd)

        return cmd

    def ensure_temp(self):
        target_db = self.config['db']
        fname = "{}/{}.gz".format(self.prefix, target_db)
        if not os.path.exists(os.path.dirname(fname)):
            os.makedirs(os.path.dirname(fname))

    def clean(self):
        if re.match(r'/$|/\*|/root[\w/]*|/etc[\w/]*|/lib[\w/]*|/usr/[\w/]*', self.prefix):
            self.log.debug("ACHTUNG! Dangerous remove try: %s", self.prefix)
            return False
        for root, dirs, files in os.walk(self.prefix, topdown=False):
            for name in files:
                os.remove(os.path.join(root, name))
            for name in dirs:
                os.rmdir(os.path.join(root, name))
        return True

    def get_util_version(self):
        cmd = self.config['binary'] + '/' + self.config['action'] + ' --version'
        try:
            result = subprocess.check_output(cmd, shell=True)
        except (subprocess.CalledProcessError, OSError) as fail:
            logging.info("Failed to get utils version. %s", fail.output)
            return None
        logging.debug("Binary version: %s", result)
        return None

    def run_cmd(self, cmd):
        try:
            result = subprocess.check_output(cmd, shell=True, stderr=subprocess.STDOUT)
            self.log.info("Command executed seccessfully. Output: %s", result)
            return True
        except (subprocess.CalledProcessError, OSError):
            self.log.error("Action %s failed.", self.config['action'])
            return False

    def dump(self, database):
        filename = database + '.gz'
        file_path = '/tmp/dump_restore/' + filename
        s3_full_path = self.s3_key_prefix + '/' + filename
        cmd = self.gen_command(database)
        is_ok = self.run_cmd(cmd)
        status = "OK" if is_ok else "FAILED"
        self.log.info("Action %s execution status for taget db %s: %s",
                      self.config['action'], database, status)
        if is_ok:
            self.log.info("Uploading backup file %s to s3://%s/%s",
                          file_path, self.s3client.bucket, s3_full_path)
            self.s3client.push(file_path, s3_full_path)
            self.log.info("Upload finished.")
        else:
            self.log.info("%s for %s failed. Skip uploading.", self.config['action'], database)

    def restore(self, database):
        filename = database + '.gz'
        file_path = '/tmp/dump_restore/' + filename
        s3_full_path = self.s3_key_prefix + '/' + filename
        self.log.info("Downloading backup file %s from s3://%s/%s",
                      file_path, self.s3client.bucket, s3_full_path)
        self.s3client.pull(s3_full_path, file_path)
        if not os.path.exists(file_path):
            self.log.error("Download backup archive %s to %s failed. Skip processing.",
                           s3_full_path, file_path)
            return None
        cmd = self.gen_command(database)
        is_ok = self.run_cmd(cmd)
        status = "OK" if is_ok else "FAILED"
        self.log.info("Action %s execution status for taget db %s: %s",
                      self.config['action'], database, status)
        return None

    def exec_action(self):
        self.clean()
        self.ensure_temp()
        for database in self.config['db'].split(','):
            if database:
                if self.config['action'] == "mongorestore":
                    self.restore(database)
                elif self.config['action'] == "mongodump":
                    self.dump(database)
                else:
                    self.log.info("Unknown action %s for db %s", self.config['action'], database)
        self.clean()
        return None
