# -*- coding: utf-8 -*-
from sandbox import sdk2
import logging
import os
from sandbox.projects.media.admins.common.utils import is_base64encoded, parse_secret
from sandbox.sandboxsdk.process import run_process
from copier import Copier, Dumper
from sandbox.sdk2 import yav


class InvalidSecretData(Exception):
    pass


class MysqlCopyDB(sdk2.Task):
    """
    Переналиваем данные MySQL серверов с помощью innobackupex или mysqldump.
    Подробнее: https://wiki.yandex-team.ru/users/chrono/CopyDatabases/
    """
    class Requirements(sdk2.Task.Requirements):
        """
        Binary and others requirements
        """
        privileged = True

    class Parameters(sdk2.Parameters):
        """
        Form parameters
        """
        donor = sdk2.parameters.String("Donor", description="Хост-источник (донор данных)")
        recipient = sdk2.parameters.String("Recipient", description="Хост-приёмник (реципиент)")
        yav_secret = sdk2.parameters.String("YAV secret with SSH key",
                                            description="sec-01d2cny7z1j[key_name]")
        switch_replication = sdk2.parameters.Bool("Switch replication",
                                                  description="Переключить репликацию"
                                                              " на мастера", default=False)
        speed_limit = sdk2.parameters.String("Speed limit",
                                             description="Ограничить скорость передачи"
                                                         " (поддерживаются 'K', 'M', 'G', 'T'"
                                                         " суффиксы)",
                                             default=0)
        with sdk2.parameters.String("Compressor", multiline=True) as compressor:
            compressor.values.zstd = compressor.Value(default=True)
            compressor.values.pzstd = None
            compressor.values.lzop = None
            compressor.values.gzip = None
            compressor.values.pigz = None
        compress_src_args = sdk2.parameters.String("Compressor arguments on source",
                                                   default="")
        compress_dst_args = sdk2.parameters.String("Compressor arguments on destination",
                                                   default="-d")
        with sdk2.parameters.String("Method", multiline=True) as method:
            method.values.xtrabackup = method.Value(default=True)
            method.values.mysqldump = None
        exclude_tables = sdk2.parameters.String("Exclude tables",
                                                description="Исключить из дампа эти таблицы"
                                                            " (только для mysqldump)")
        mysql_secret = sdk2.parameters.String("MySQL client secret",
                                              description="sec-01d2cny7z1j[user_name]"
                                                          " secret value = password")
        dbname = sdk2.parameters.String("DB name", description="Имя базы")

    log = logging.getLogger("XtraCopySRV")

    @staticmethod
    def prepare():
        update = ['sudo', 'apt-get', 'update']
        install = ['sudo', 'apt-get', 'install', '--yes', '--force-yes', 'lzop', 'zstd']
        run_process(update)
        run_process(install)

    def on_execute(self):

        self.prepare()
        donor = self.Parameters.donor
        recipient = self.Parameters.recipient
        self.log.debug("Execute task with Parameters: %s", self.Parameters.__dict__)
        self.log.info("Do sync DB donor: %s, recipient: %s", donor, recipient)
        import base64

        _sec, _key = parse_secret(self.Parameters.yav_secret)
        key = yav.Secret(_sec).data()[_key]

        # key = yav.Secret(self.Parameters.yav_secret).data()[self.Parameters.yav_secret_keyname]
        if is_base64encoded(key):
            self.log.debug("Got base64 encoded data. Try to decode it.")
            key = base64.b64decode(key)
        if not key:
            raise InvalidSecretData("Can't get valid SSH key from secret %s key %s", _sec, _key)

        key_path = str(self.path()) + '/key.rsa'
        with open(key_path, 'w+') as kfile:
            kfile.write(key)
        os.chmod(key_path, 0600)
        compressor = self.Parameters.compressor
        compress_src_args = self.Parameters.compress_src_args
        compress_dst_args = self.Parameters.compress_dst_args
        if self.Parameters.method == 'mysqldump':
            if not self.Parameters.dbname:
                self.log.error("Can not process mysqldump without database name. Exit.")
                raise Exception("Migration unavailable with method mysqldump without db name.")
            if self.Parameters.mysql_secret:
                _sec, _key = parse_secret(self.Parameters.mysql_secret)
                mysql_user = _key
                mysql_password = yav.Secret(_sec).data()[mysql_user]

                if is_base64encoded(mysql_password):
                    self.log.debug("Got base64 encoded data. Try to decode it.")
                    mysql_password = base64.b64decode(mysql_password)
                if not mysql_password:
                    raise self.log.info("Can't get valid data from secret %s key %s", _sec, _key)
            else:
                mysql_user = None
                mysql_password = None
            cp = Dumper(mysql_user, mysql_password, self.Parameters.exclude_tables,
                        self.Parameters.dbname, donor, recipient, key_path,
                        compressor, compress_src_args, compress_dst_args,
                        self.Parameters.speed_limit)
            if not cp.check_id():
                raise Exception("Can't run command on target hosts. See debug log for details.")
            cp.run_mysqldump()
        else:
            cp = Copier(donor, recipient, key_path, compressor,
                        compress_src_args, compress_dst_args, self.Parameters.speed_limit)
            if not cp.check_id():
                raise Exception("Can't run command on target hosts. See debug log for details.")
            cp.migrate_data()
            if self.Parameters.switch_replication:
                cp.switch_master()
