from __future__ import division
import logging
import os
import subprocess
from time import time, sleep
import socket
from sandbox.projects.media.admins.common.utils import obfuscate
import paramiko


class GTIDException(Exception):
    pass


class CopyDataException(Exception):
    pass


class MasterInfoException(Exception):
    pass


class Client(object):
    def __init__(self, host, key, **kwargs):
        self._client = paramiko.SSHClient()
        self._client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        self.host = host
        self.log = logging.getLogger("SSHClient")
        self.log.debug("Set private key with file %s", key)
        self.key = paramiko.RSAKey.from_private_key_file(key)
        self.log.debug("Private key set.")
        if 'username' in kwargs:
            self._username = kwargs.__getitem__('username')
        else:
            self._username = 'robot-cult'

    def connect(self):
        # check if connection is inactive
        if not self._client.get_transport():
            self.log.debug("Connecting to %s", self.host)
            _start_ts = time()
            try:
                self._client.connect(hostname=self.host, pkey=self.key,
                                     username=self._username, banner_timeout=30)
                _op_time = time() - _start_ts
                self.log.debug("Connection success in %s sec", _op_time)
            except paramiko.SSHException as err:
                _op_time = time() - _start_ts
                self.log.debug("Connection to %s with key %s failed. Timing: %s",
                               self.host,
                               obfuscate(self.key.get_base64(), 5, 5),
                               _op_time
                               )
                self.log.error(err)

    def run(self, cmd, timeout=None):
        """
        It just runs command
        :param cmd: command to run
        :param timeout: timeout for command
        :return: return transport objects with stdin stdout stderr
        """
        if not self._client.get_transport():
            self.connect()
        _start_ts = time()
        self.log.debug("%s: Run command %s", self.host, cmd)
        try:
            conn = self._client.exec_command(cmd, timeout=timeout)
            _op_time = time() - _start_ts
            self.log.debug("Remote command run success. op. time: %s", _op_time)
            return conn
        except (Exception, socket.timeout) as err:
            _op_time = time() - _start_ts
            self.log.debug("Failed to run %s on %s. op. time: %s", cmd, self.host, _op_time)
            self.log.error(err)

        return None, None, err

    def exec_command(self, cmd, timeout=None):
        """
        It does the same as run() but provides more control
        :param cmd: command to run
        :param timeout: timeout to exec command
        :return: tuple with paramiko session object and stderr stream
        """
        if not self._client.get_transport():
            self.connect()

        sess = None
        _start_ts = time()
        try:
            self.log.debug("Exec command %s", cmd)
            sess = self._client.get_transport().open_channel("session")
            stderr = sess.makefile_stderr()
            sess.settimeout(timeout)
            sess.exec_command(cmd)
            _op_time = time() - _start_ts
            self.log.debug("Command %s succeed; time: %s", cmd, _op_time)
        except (paramiko.SSHException, socket.timeout) as err:
            self.log.error("Failed to create ssh sesion: %s", err)
            stderr = err

        return sess, stderr

    def channel_rl(self, ch):
        try:
            return ch.readline().decode('utf-8').strip()
        except (paramiko.SSHException, socket.timeout, Exception) as err:
            self.log.debug("Can't read channel data: %s", err)
            return


class Copier(object):
    def __init__(self, donor, recipient, key, compressor,
                 compress_src_args, compress_dst_args, speed):
        self.donor = donor
        self.recipient = recipient
        self.key = key
        self.log = logging.getLogger(self.__class__.__name__)
        self.ssh_prefix = ['ssh', '-i', self.key]
        self.user = 'robot-cult'
        self.compressor = compressor
        self.compress_src_args = compress_src_args
        self.compress_dst_args = compress_dst_args
        self.cluster_master = self.donor
        self._rec_ssh = Client(self.recipient, self.key)
        self._donor_ssh = Client(self.donor, self.key)
        self.set_cluster_master()
        self._speed_limit = speed

    def remote_run(self, host, cmd, retry=3):
        ssh = self.ssh_prefix + ['{}@{}'.format(self.user, host)]
        self.log.debug("Run on %s command: %s", host, cmd)
        command = ssh + cmd.split()
        returncode, out, err = (1, None, None)
        while retry > 0 and returncode != 0:
            sp = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            out, err = sp.communicate()
            returncode = sp.wait()
            msg = "Command: {} exited on {}. Result: rc={}, stdout={}, stderr={}".format(
                cmd, host, returncode, out, err)
            self.log.debug(msg)
            self.log.debug("Command output: %s; stderr: %s", out, err)
            retry -= 1
            if returncode != 0:
                self.log.debug("Exitcode: %s, stderr: %s. Try again.", returncode, err)

        return returncode, out, err

    def run_async(self, host, cmd):
        import multiprocessing
        self.log.info('Host %s: RUN ASYNC command: %s', host, cmd)
        proc = multiprocessing.Process(target=self.remote_run, args=[host, cmd])
        return proc

    def get_data_size(self):
        _, out, err = self._donor_ssh.run("du -bs /opt/mysql | awk '{print $1}'")
        # rc, out, err = self.remote_run(host, "du -bs /opt/mysql | awk '{print $1}'")
        try:
            return int(self._donor_ssh.channel_rl(out))
        except ValueError:
            self.log.debug("Got empty output.")
            return 0

    def set_cluster_master(self):
        _, out, _ = self._donor_ssh.run("sudo /usr/bin/mysql"
                                        " --defaults-file=/etc/mysql/client.cnf -Bn"
                                        " -e \'show slave status\\G;\'"
                                        " | grep Master_Host: | awk \'{print $2;}\'"
                                        )

        master = out.readline().decode("utf-8").strip()

        if not master:
            master = self.donor

        self.log.info("Set master host to %s", master)
        self.cluster_master = master

    def prepare_recipient(self):
        self._rec_ssh.run("sudo killall -9 nc.openbsd || true")
        # self.remote_run(self.recipient, 'sudo killall -9 nc.openbsd || true')

        sess, err = self._rec_ssh.exec_command("which nc.openbsd")
        if not sess:
            return False
        rc = sess.recv_exit_status()
        nc = sess.recv(1024).decode('utf-8').strip()
        # rc, nc, _ = self.remote_run(self.recipient, 'which nc.openbsd')
        if rc > 0 or not nc:
            self.log.error("nc.openbsd not found on %s. Install netcat-openbsd.", self.recipient)
            return False
        cmd_set = [
            'touch /tmp/mysql_refill.tmp',
            'sudo /etc/init.d/mysql stop',
            'sudo rm -rf /opt/mysql',
            'sudo mkdir /opt/mysql',
            'sudo chown mysqladmin /opt/mysql'
        ]
        for cmd in cmd_set:
            _, out, err = self._rec_ssh.run(cmd)
            out = self._rec_ssh.channel_rl(out)
            err = self._rec_ssh.channel_rl(err)
            self.log.debug("%s: Executed %s, out: %s, err: %s", self.recipient, cmd, out, err)

    def is_gtid_on(self):
        cmd = "sudo grep gtid_mode /etc/mysql/my.cnf | awk '{print $NF}'"
        _, gtid_mode, _ = self._donor_ssh.run(cmd)
        if gtid_mode:
            gtid_mode = self._donor_ssh.channel_rl(gtid_mode)
        self.log.info('Got gtid_mode %s', gtid_mode)
        if gtid_mode == 'ON':
            return True
        if gtid_mode == 'OFF':
            return False
        raise GTIDException('Invalid gtid_mode on recipient `%s`', self.recipient)

    def update_grants(self):
        self.log.info("Update mysql-grants.")
        # rc, _, _ = self.remote_run(self.recipient, "sudo /usr/bin/mysql-grants-update-4")
        _, _, err = self._rec_ssh.run("sudo /usr/bin/mysql-grants-update-4")
        self.log.info("mysql-grants-update-4 exited with stderr: %s",
                      err.read().decode('utf-8').strip())

    def check_id(self):
        cmd = '/usr/bin/id {}'.format(self.user)
        sess, err = self._rec_ssh.exec_command(cmd)
        if not sess:
            return False
        while not sess.exit_status_ready():
            self.log.debug("Waiting for check...")
            sleep(1)

        if sess.recv_exit_status() != 0:
            self.log.debug("Cant't get user info for %s: err: %s",
                           self.user, self._rec_ssh.channel_rl(err)
                           )
            return False

        return True

    def migrate_data(self):
        from datetime import timedelta
        # todo: move copy data logic into separate method and implement retries
        self.prepare_recipient()
        self.log.info("Start database copy from %s to %s", self.donor, self.recipient)
        size = self.get_data_size()
        self.log.debug("Got data size: %s bytes.", size)
        cmd = 'cd /opt/mysql; set -o pipefail; nc.openbsd -v -d -6 -l 13456 |' \
              ' {} {} | pv -n -s {} --rate-limit {}' \
              ' | sudo ionice -c 3 tar ix'.format(self.compressor, self.compress_dst_args,
                                                  size, self._speed_limit)
        self.log.info("Run on %s with separate thread command %s", self.recipient, cmd)
        ts = time()
        rec_sess, rec_err = self._rec_ssh.exec_command(cmd)
        if not rec_sess:
            return False

        if self.cluster_master == self.donor:
            cmd = 'cd /opt/mysql; sleep 5; set -o pipefail; sudo -H innobackupex' \
                  ' --stream=tar ./ | {} {} |' \
                  ' nc.openbsd -q 0 -v {} 13456'.format(self.compressor,
                                                        self.compress_src_args,
                                                        self.recipient)
        else:
            cmd = 'cd /opt/mysql; sleep 5; set -o pipefail; sudo -H innobackupex' \
                  ' --slave-info --stream=tar ./ | {} {} |' \
                  ' nc.openbsd -q 0 -v {} 13456'.format(self.compressor,
                                                        self.compress_src_args,
                                                        self.recipient)

        self.log.info("Run on %s command %s", self.donor, cmd)
        donor_sess, donor_err = self._donor_ssh.exec_command(cmd)
        if not donor_sess:
            return False

        # wait for data copy and check
        sleep_time = 5
        while not rec_sess.exit_status_ready():
            timeout = sleep_time if sleep_time > 5 else None
            out = None
            try:
                _, out, err = self._rec_ssh.run("du -bs /opt/mysql", timeout=timeout)
            except Exception as err:
                self.log.debug("Failed to get progress stats: %s", err)

            if out:
                out = self._rec_ssh.channel_rl(out)
                if out:
                    done_size = int(out.split()[0])
                else:
                    done_size = 1
            else:
                done_size = 1
                self.log.info("Can't get current data size. continue...")

            if done_size != size and done_size > 0:
                elapsed_time = time() - ts
                speed = done_size / elapsed_time
                remain_time = round((size - done_size) / speed, 0)
            elif done_size < 0:
                speed = 0
                remain_time = 3600
            else:
                elapsed_time = time() - ts
                speed = done_size / elapsed_time
                remain_time = 0
            if 10 < remain_time < 100:
                sleep_time = round((remain_time / 10), 0)
            elif 100 < remain_time < 10000:
                sleep_time = round((remain_time / 100), 0)
            else:
                sleep_time = 1
            prc = round(done_size * 100 / size, 2)
            self.log.debug("Got remain time %s", remain_time)
            msg = 'Calculated stats: processed {}% ({}/{}) MB, speed {} MB/sec, ' \
                  'ETA {}, sleep_time for next info {} sec'
            try:
                eta = timedelta(seconds=remain_time).__str__()
            except OverflowError:
                eta = remain_time
                self.log.debug("Failed to convert %s to timedelta, use 3600", remain_time)
            msg = msg.format(
                prc,
                round((done_size / (1024 * 1024)), 3),
                round((size / (1024 * 1024)), 3),
                round((speed/1024/1024), 2),
                eta,
                sleep_time
            )
            self.log.info(msg)
            sleep(sleep_time)

        self.log.info("Command finished with rc: %s, stderr: %s, stdout: %s",
                      rec_sess.recv_exit_status(),
                      self._rec_ssh.channel_rl(rec_err),
                      self._rec_ssh.channel_rl(rec_sess)
                      )

        if rec_sess.recv_exit_status() != 0:
            self.log.error("%s: cmd: %s, out: %s, err: %s",
                           self.donor,
                           cmd,
                           self._donor_ssh.channel_rl(donor_sess),
                           self._donor_ssh.channel_rl(donor_err)
                           )
            self.log.error("%s: nc.openbsd listener failed, out: %s, err: %s",
                           self.recipient,
                           self._rec_ssh.channel_rl(rec_sess),
                           self._rec_ssh.channel_rl(rec_err)
                           )
            raise CopyDataException("Data copying failed.")

        # let the global task timeout to handle it
        # atempts = 60
        # while atempts > 0:
        while not self.check_id():
            self.log.info("Waiting for %s comes back to normal state(see debug log for details)",
                          self.recipient)
            sleep(1)

        cmd = "sleep 5; cd /opt/mysql; sudo innobackupex --apply-log" \
              " --ibbackup=`ls /usr/bin/xtrabackup* | sort -r | head -1 | cut -f4 -d/` ./"

        self.log.info("Run innobackupex --apply-log")
        rec_sess, rec_err = self._rec_ssh.exec_command(cmd)
        inter = 5
        while not rec_sess.exit_status_ready():
            self.log.debug("Waiting fow innobackupex --apply-log...")
            sleep(inter)
        self.log.info("Command finished with rc: %s, stderr: %s",
                      rec_sess.recv_exit_status(), self._rec_ssh.channel_rl(rec_err)
                      )

        self._rec_ssh.run("sudo chown -R mysql:mysql /opt/mysql")
        self.log.info("Start mysql server.")
        rec_sess, rec_err = self._rec_ssh.exec_command("sudo /usr/sbin/service mysql start")
        inter = 1
        while not rec_sess.exit_status_ready():
            self.log.info("Starting mysql server...")
            sleep(inter)

        while not donor_sess.exit_status_ready():
            self.log.info("Waiting for donor process ends...")

        self.log.debug("%s: command exited with rc: %s, stderr: ",
                       self.donor, donor_sess.recv_exit_status(),
                       self._donor_ssh.channel_rl(donor_err)
                       )
        if rec_sess.recv_exit_status() > 0:
            self.log.error("%s: command finished with rc: %s, stderr: %s",
                           self.recipient,
                           rec_sess.recv_exit_status(), self._rec_ssh.channel_rl(rec_err)
                           )
            raise Exception("Can't start mysql server on %s. See log for details.", self.recipient)
        else:
            self.log.info("%s command finished with rc: %s, stderr: %s",
                          self.recipient,
                          rec_sess.recv_exit_status(),
                          self._rec_ssh.channel_rl(rec_err)
                          )

        self.update_grants()

        self.log.debug("Remove temporary refill file")
        self._rec_ssh.run('rm /tmp/mysql_refill.tmp')

    def switch_master(self):
        import re
        self.log.info("Switching replication on %s to master_host %s.", self.recipient,
                      self.cluster_master)
        self._rec_ssh.run("sudo chown -R mysql:mysql /opt/mysql")
        self._rec_ssh.run("killall -9 nc.openbsd || true")
        self._rec_ssh.run("sudo /etc/init.d/mysql start")
        cmd = "sudo my_print_defaults --defaults-file=/etc/mysql/client.cnf" \
              " --show mysql-purge-binlogs | grep password | awk -F= '{print $2}'"
        _, pwd, _ = self._rec_ssh.run(cmd)
        if pwd:
            pwd = self._rec_ssh.channel_rl(pwd)
        self.log.info("Got password %s", obfuscate(pwd, 0, 3))
        cmd = "sudo my_print_defaults --defaults-file=/etc/mysql/client.cnf" \
              " --show mysql-purge-binlogs | grep user | awk -F= '{print $2}'"
        _, user, _ = self._rec_ssh.run(cmd)
        if user:
            user = self._rec_ssh.channel_rl(user)
            user = user.strip()
        else:
            user = None
        self.log.info("Got replication user %s", user)
        cmd = "my_print_defaults --defaults-file=/etc/mysql/my.cnf mysqld" \
              " | grep \'\-port\' | awk -F= '{print $2}'"
        _, port, _ = self._rec_ssh.run(cmd)
        port = self._rec_ssh.channel_rl(port)
        self.log.info("Got port %s", port)

        cmd = "sudo sed -re ':a;N;$!ba;s/,(\\n|\s)/,/g' " \
              "/opt/mysql/xtrabackup_binlog_info | awk '{print $3}'"
        _, gtid_purged, _ = self._rec_ssh.run(cmd)
        if gtid_purged:
            gtid_purged = self._rec_ssh.channel_rl(gtid_purged)
        self.log.info("Got gtid_purged `%s`", gtid_purged)

        # GTID is ON (new)
        #
        if gtid_purged and self.is_gtid_on():
            if self.cluster_master == self.donor:
                master_host = self.donor
            else:
                master_host = self.cluster_master

            cmd = "sudo mysql --defaults-file=/etc/mysql/client.cnf -e \"reset master; reset slave; "\
                  "set global gtid_purged=\'{}\'; "\
                  "change master to "\
                  "master_host=\'{}\', master_port={}, master_auto_position=1, " \
                  "master_user=\'{}\', master_password=\'{}\', master_ssl=0; start slave;\"".format(
                      gtid_purged, master_host, int(port), user, pwd)

            # cmd
            self.log.info("Execute change master to statement on %s", self.recipient)
            self._rec_ssh.run(cmd)

        # GTID is OFF (old behaviour)
        #
        else:
            if self.cluster_master == self.donor:
                cmd = "sudo cat /opt/mysql/xtrabackup_binlog_info"
                _, binlog_info, _ = self._rec_ssh.run(cmd)
                log_file, log_pos = self._rec_ssh.channel_rl(binlog_info).split()
                log_pos = log_pos.strip()
                log_file = log_file.strip()
            else:
                cmd = "sudo cat /opt/mysql/xtrabackup_slave_info"
                _, binlog_info, _ = self._rec_ssh.run(cmd)
                if not binlog_info:
                    raise MasterInfoException("Failed to read binlog information.")

                binlog_info = self._rec_ssh.channel_rl(binlog_info)
                log_file = re.findall(r"MASTER_LOG_FILE=\'([\w\d.\-]+)\'", binlog_info)[0]
                log_pos = re.findall(r"MASTER_LOG_POS=([\d]+)", binlog_info)[0]

            self.log.info("Got binlog info: log_file=%s, log_pos=%s", log_file, log_pos)
            cmd = "sudo mysql --defaults-file=/etc/mysql/client.cnf -e \"change master to " \
                  "master_host=\'{}\', master_port={}, master_log_pos={}, master_log_file=\'{}\'," \
                  "master_user=\'{}\', master_password=\'{}\', master_ssl=0;\"".format(
                      self.cluster_master, int(port), int(log_pos), log_file, user, pwd)

            # cmd
            self.log.info("Execute change master to statement on %s", self.recipient)
            self._rec_ssh.run(cmd)

        # self.log.info("Start slave on %s", self.recipient)
        # cmd = "sudo mysql --defaults-file=/etc/mysql/client.cnf -e \"start slave;\""
        # self._rec_ssh.run(cmd)
        slave_status = self.check_slave()
        if slave_status:
            msg = "Slave_IO_Running: {}, Slave_SQL_Running: {}, Slave_IO_State: {}, Lag: {}".format(
                slave_status['Slave_IO_Running'],
                slave_status['Slave_SQL_Running'],
                slave_status['Slave_IO_State'],
                slave_status['Seconds_Behind_Master']
            )
            self.log.info(msg)

    def check_slave(self):
        cmd = "sudo mysql --defaults-file=/etc/mysql/client.cnf -e \"show slave status\G;\""
        _, _out, _ = self._rec_ssh.run(cmd)
        if _out:
            try:
                st = _out.read().splitlines()
            except (paramiko.SSHException, socket.timeout, Exception):
                self.log.warning("Failed to get slave status")
                return None

            if not st:
                self.log.info("No slave info, may be %s is master.", self.recipient)
                return None
            status = {}
            try:
                st.pop(0)
                for line in st:
                    _k, _v = line.strip().split(":", 1)
                    status.update({_k.strip(): _v.strip()})
            except (ValueError, IndexError, TypeError):
                self.log.warning("Failed to convert slave status output. Raw output: %s", st)
                return None

            self.log.debug(status)
            return status

        return None

    def clean_prcesses(self):
        self._rec_ssh.run("sudo killall -9 nc.openbsd || true")
        self._donor_ssh.run("sudo killall -9 nc.openbsd || true")


class Dumper(Copier):
    def __init__(self, user, password, exclude, dbname, donor, recipient, key, compressor,
                 compress_src_args, compress_dst_args, speed):
        super(Dumper, self).__init__(donor, recipient, key, compressor, compress_src_args,
                                     compress_dst_args, speed)
        if not user or not password:
            self.defaults_file = '/root/.my.cnf'
        else:
            self.mysql_user = user
            self.password = password
            self.defaults_file = '/tmp/.tmp.my.cnf'
            self.prepare_defaults()

        self.local_mysql = "sudo /usr/bin/mysql --defaults-file=/root/.my.cnf -BN"
        self.target_mysql = "/usr/bin/mysql --defaults-file={}" \
                            " --init-command=\'SET FOREIGN_KEY_CHECKS=0;\'" \
                            " --host={} --port={} -A".format(self.defaults_file,
                                                             self.recipient, 3306)

        self.exclude_tables = set(exclude.split(',')) if exclude is not None else set()
        self.target_db = dbname

    def remote_run(self, host, cmd, retry=3):
        import re
        ssh = self.ssh_prefix + ['{}@{}'.format(self.user, host)]
        self.log.debug("Run on %s command: %s", host, cmd)
        command = ssh + cmd.split()
        returncode, out, err = (1, None, None)
        while retry > 0 and returncode != 0:
            sp = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            out, err = sp.communicate()
            returncode = sp.wait()
            msg = "Command: {} exited on {}. Result: rc={}, stdout={}, stderr={}".format(
                cmd, host, returncode, out, err)
            self.log.debug(msg)
            self.log.debug("Command output: %s; stderr: %s", out, err)
            retry -= 1
            if returncode != 0 or re.findall(r'mysqldump:\s[Ee]rror\s\d+', err):
                self.log.debug("Exitcode: %s, stderr: %s. Lets try again.", returncode, err)
            else:
                self.log.debug("Command exitcode: %s", returncode)

        return returncode, out, err

    def prepare_defaults(self):
        self.log.debug("Prepare defaults mysql file.")
        path = os.path.dirname(self.key) + '/.tmp.my.cnf'
        with open(path, 'w') as df:
            df.write(u"[client]\nuser={}\npassword={}\n".format(self.mysql_user, self.password))
        cmd = "scp -i {} {} {}@{}:{}".format(self.key, path, self.user,
                                             self.donor, self.defaults_file)
        sp = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = sp.communicate()
        returncode = sp.wait()
        self.log.debug("Preparing defaults-file result: out %s, err %s, rc %s",
                       out, err, returncode)

    def clean_defaults(self):
        self.log.debug("Clean temp defaults")
        cmd = "sudo rm -f {}".format(self.defaults_file)
        self.remote_run(self.donor, cmd)

    def get_table_size(self, host, db, table):
        """return table size in bytes"""
        query = "SELECT (data_length + index_length) FROM" \
                " information_schema.TABLES WHERE table_schema = \'{}\'" \
                " AND table_name = \'{}\';".format(db, table)
        if host == self.donor:
            cmd = self.local_mysql + " {} -e \"{}\"".format(db, query)
        else:
            cmd = self.target_mysql + " -BN {} -e \"{}\"".format(db, query)

        _, size, _ = self.remote_run(self.donor, cmd)
        size = size.strip()
        if not size:
            size = 0

        return int(size)

    def dump_table(self, table):
        db = self.target_db
        mysqldump = "sudo /usr/bin/mysqldump --defaults-file=/root/.my.cnf"
        table_size = self.get_table_size(self.donor, db, table)
        self.log.debug("Got table size: %s bytes", table_size)
        cmd = "{} --add-drop-table --lock-for-backup --single-transaction {} {}" \
              " | {} {}".format(mysqldump, db, table, self.target_mysql, db)
        self.log.debug("Run command on %s: %s", self.donor, cmd)
        self.log.info("Start import table %s.", table)
        ts = time()
        sleep_time = 1
        self.log.info("Start migrate table %s data.", table)
        proc = self.run_async(self.donor, cmd)
        proc.start()
        sleep(sleep_time)
        while proc.is_alive():
            done_size = self.get_table_size(self.recipient, db, table)
            if not done_size:
                done_size = 0
            prc = round(done_size * 100 / table_size, 2)
            msg = "Processed data {}% ({}/{})bytes ({}/{})MB".format(
                prc, done_size, table_size, round((done_size / (1024 * 1024)), 3),
                round((table_size / (1024 * 1024)), 3))
            self.log.info(msg)
            if done_size != table_size and done_size > 0:
                elapsed_time = time() - ts
                speed = done_size / elapsed_time
                remain_time = round((table_size - done_size) / speed, 0)
            elif done_size < 0:
                speed = 0
                remain_time = 100500
            else:
                elapsed_time = time() - ts
                speed = done_size / elapsed_time
                remain_time = 0
            if 10 < remain_time < 100:
                sleep_time = round((remain_time / 10), 0)
            elif 100 < remain_time < 1000:
                sleep_time = round((remain_time / 100), 0)
            else:
                sleep_time = 1
            msg = 'Calculated stats for {}: speed {} MB/sec, ETA {} sec,' \
                  ' sleep_time for next info {} sec'.format(table, round((speed/1024/1024), 2),
                                                            remain_time, sleep_time)
            self.log.info(msg)
            sleep(sleep_time)

        rc = proc.exitcode
        self.log.debug("Child exitcode is: %s", rc)
        done_ts = time() - ts
        if rc != 0:
            self.log.info("Migration for table %s failed.", table)
        else:
            self.log.info("Migration for table %s done. Time: %s sec", table, done_ts)

    def run_mysqldump(self):
        db = self.target_db
        cmd = self.local_mysql + " -e \'show tables in {};\'".format(db)
        _, tables, _ = self.remote_run(self.donor, cmd)
        self.log.debug("Got tables form %s: %s", self.donor, tables)
        tables = set(tables.split()) - self.exclude_tables
        if self.exclude_tables:
            self.log.info("Tables %s will be skipped.", self.exclude_tables)
        for table in tables:
            self.dump_table(table)
