import logging
from sandbox import sdk2
import subprocess
import sandbox.common.types.client as ctc
from sandbox.sdk2 import yav


class ConfigException(Exception):
    pass


class PgToYT(sdk2.Task):
    """
    Import PostgreSQL databases tables data into YT
    Simple task which just runs pg_to_yt from Transfer Manager
    https://wiki.yandex-team.ru/users/chrono/Impotrt-dannyx-iz-PGAAS-v-YT/
    https://wiki.yandex-team.ru/transfer-manager/copy/pgtoyt/
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        cores = 2
        ram = 4  # 4GiB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        config = sdk2.parameters.JSON('Import config', required=True, description='Import config')

    log = logging.getLogger('PgToYT')

    def on_prepare(self):
        self.validate_config()
        self.log.info('Prepare resource PG_TO_YT_EXECUTABLE')
        resource = sdk2.Resource['PG_TO_YT_EXECUTABLE'].find()
        if resource.count > 0:
            res = resource.first()
            binary = sdk2.ResourceData(res)
            self.Context.pg_to_yt = str(binary.path)
            self.Context.save()
        else:
            self.log.error('PG_TO_YT_EXECUTABLE not found. Try to build it(task id = 529854153)')

    def get_bin(self):
        import os
        os.environ['PATH'] += ':{}'.format(self.Context.pg_to_yt)
        return self.Context.pg_to_yt

    def validate_config(self):
        self.log.info('Validate config.')
        try:
            config = dict(self.Parameters.config)
        except ValueError:
            raise ConfigException('Invalid config. Please check import config(JSON)')

        _yt_sec = False
        if 'yt_secret' in config:
            _s = config['yt_secret']
            try:
                _sec, _key = _s.replace(' ', '').split(':')
                _yt_sec = True
            except ValueError:
                raise ConfigException('Incorrect global secret definition: %s' % _s)
            if not _sec or not _key:
                raise ConfigException('Incorrect global secret definition: %s' % _s)

        for job in config['jobs']:
            task = config['jobs'][job]
            if 'pg_secret' not in task:
                raise ConfigException('PG secret not defined for job %s' % job)
            try:
                _sec, _key = task['pg_secret'].replace(' ', '').split(':')
            except ValueError:
                raise ConfigException('Invalid job in config: %s' % job)
            if not _sec or not _key:
                raise ConfigException('Incorrect PG secret definition for job %s' % job)

            if 'yt_secret' not in task and not _yt_sec:
                raise ConfigException('YT secret not defined for job %s' % job)
            if 'yt_secret' in task:
                try:
                    _sec, _key = task['yt_secret'].replace(' ', '').split(':')
                except ValueError:
                    raise ConfigException('Incorrect YT secret definition for job %s' % job)
                if not _sec or not _key:
                    raise ConfigException('Incorrect YT secret definition for job %s' % job)

        self.log.info('Config validation is successfully completed.')
        return True

    def on_execute(self):
        config = dict(self.Parameters.config)
        self.log.debug(config)
        pg_to_yt = self.get_bin()
        _yt_token = None
        _db_schema = None

        # if used global yt secret
        if 'yt_secret' in config:
            self.log.info('Use global YT secret.')
            _sec, _key = config['yt_secret'].replace(' ', '').split(':')
            _yt_token = yav.Secret(_sec).data()[_key]

        for job in config['jobs']:
            task = config['jobs'][job]
            self.log.info('Execute import task %s: %s', job, task)
            _sec, _key = task['pg_secret'].replace(' ', '').split(':')
            _pg_password = yav.Secret(_sec).data()[_key]

            _db_schema = task.get('db_schema', 'public')
            self.log.info('Use db schema: ' + str(_db_schema))

            # override global yt secret if needed
            if 'yt_secret' in task:
                self.log.info('Use per task(overrided) YT secret for task %s', job)
                _sec, _key = task['yt_secret'].replace(' ', '').split(':')
                _yt_token = yav.Secret(_sec).data()[_key]

            if not _yt_token:
                raise Exception('Can\'t get YT secret. Check config and secret please.')

            cmd = [
                pg_to_yt,
                '-db-host', task['db_host'],
                '-db-name', task['db_name'],
                '-db-user', task['db_user'],
                '-yt-cluster', task['yt_cluster'],
                '-yt-path', task['yt_path'],
                '-yt-token', _yt_token,
                '-db-password', _pg_password
            ]

            if _db_schema:
                cmd.extend(('-db-schema', _db_schema))

            if 'tables' in task:
                self.log.info('Import only specified tables: %s.', task['tables'])
                for table in task['tables']:
                    cmd.extend(('-db-table', table))
            else:
                self.log.info('Import whole database %s.', task['db_name'])

            try:
                sp = subprocess.Popen(cmd, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
                out, err = sp.communicate()
                rc = sp.wait()
                self.log.info('RC: %s, Out: %s, Err: %s', rc, out, err)
            except subprocess.CalledProcessError as err:
                self.log.error(err)
                raise
