import logging
import os
import requests
from zipfile import ZipFile
from datetime import datetime
from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm


class InvalidSecretData(Exception):
    pass


class S3Backup(sdk2.Task):
    """ Backup S3 buckets """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        client_tags = ctc.Tag.LINUX_BIONIC
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        yav_aws_s3_secret = sdk2.parameters.YavSecret('YAV secret with AWS S3 keys',
                                                      description='Should have "access_key" and "secret_key" keys',
                                                      required=True)
        is_production_s3 = sdk2.parameters.Bool('Use production S3', required=True, default=False)
        src_s3 = sdk2.parameters.String('Source S3', required=True)
        dst_s3 = sdk2.parameters.String('Destination S3', required=True)

    @staticmethod
    def prepare():

        def extract_all_with_permission(zf, path=os.getcwd()):
            zip_unix_system = 3
            for info in zf.infolist():
                extracted_path = zf.extract(info, path)
                if info.create_system == zip_unix_system:
                    unix_attributes = info.external_attr >> 16
                    if unix_attributes:
                        os.chmod(extracted_path, unix_attributes)

        url = 'https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip'
        dl_file = os.environ['HOME'] + '/awscliv2.zip'

        # Download AWS CLI
        logging.info('Downloading AWS CLI')
        r = requests.get(url)
        with open(dl_file, 'wb') as f:
            f.write(r.content)

        # Unzip AWS CLI
        logging.info('Unzipping AWS CLI')
        with ZipFile(dl_file, 'r') as f:
            extract_all_with_permission(f, os.environ['HOME'])

        # Install AWS CLI
        logging.info('Installing AWS CLI')
        step_install = ['/bin/bash',
                        os.environ['HOME'] + '/aws/install',
                        '--install-dir', os.environ['HOME'] + '/aws-cli',
                        '--bin-dir', os.environ['HOME'] + '/aws-cli_symlink']
        run_process(step_install)

        # Check AWS CLI
        logging.info('Checking AWS CLI')
        step_check = [os.environ['HOME'] + '/aws-cli_symlink/aws', '--version']
        run_process(step_check)

    def on_execute(self):
        # Log parameters
        logging.info('YAV secret with AWS S3 config: {}'.format(self.Parameters.yav_aws_s3_secret))
        logging.info('Use production S3: {}'.format(self.Parameters.is_production_s3))
        endpoint_url = 'https://s3.mds.yandex.net' if self.Parameters.is_production_s3 else 'https://s3.mdst.yandex.net'
        logging.info('AWS endpoint URL: {}'.format(endpoint_url))
        logging.info('Source S3: {}'.format(self.Parameters.src_s3))
        logging.info('Destination S3: {}'.format(self.Parameters.dst_s3))
        backup_dir = self.Parameters.dst_s3 + '/' + datetime.today().strftime('%Y%m%d_%H%M%S')
        logging.info('Backup S3 directory: {}'.format(backup_dir))

        # Preparing
        logging.info('Preparing')
        self.prepare()
        logging.info('Prepare done')

        # Get AWS S3 config from YAV
        logging.info('Getting YAV secret')
        yav_aws_s3 = self.Parameters.yav_aws_s3_secret.data()
        aws_conf_path = os.environ['HOME'] + '/.aws/credentials'
        dir_path = os.path.dirname(aws_conf_path)
        if not os.path.exists(dir_path):
            os.mkdir(dir_path)
        with open(aws_conf_path, 'w') as f:
            f.write('[default]\n')
            f.write('aws_access_key_id = {}\n'.format(yav_aws_s3['access_key']))
            f.write('aws_secret_access_key = {}\n'.format(yav_aws_s3['secret_key']))
        os.chmod(aws_conf_path, 0600)

        # Perform backup
        logging.info('Performing backup')
        backup_cmd = [os.environ['HOME'] + '/aws-cli_symlink/aws',
                      '--profile=default',
                      '--endpoint-url={}'.format(endpoint_url),
                      '--no-verify-ssl',
                      's3',
                      'cp',
                      self.Parameters.src_s3,
                      backup_dir,
                      '--recursive',
                      '--no-progress']
        logging.info('AWS cmd: {}'.format(backup_cmd))
        run_process(backup_cmd)
