# -*- coding: utf-8 -*-

"""
    DESCRIPTION: Sandbox task for building yaml spec from jsonnet for Y.D.
    AUTHOR: @coldmind
    ST_QUEUE: CADMIN
"""


import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

import re
import logging
import yaml
import time

from os import path

from sandbox import common
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sdk2.helpers import subprocess, ProcessLog
from sandbox.projects.media.utils.secret_helper import SecretHelper


class JsonnetToDeployParameters(sdk2.Parameters):
    container = sdk2.parameters.Container('Container', required=True,
                                          platform='linux_ubuntu_18.04_bionic')

    with sdk2.parameters.Group('Jsonnet to deploy parameters') as group_parameters:
        path_to_jsonnet = sdk2.parameters.String(
            'Path to jsonnet',
            description='Arcadia path (after /arc/trunk/arcadia/) to .jsonnet file',
            default_value='kinopoisk/deployments/services/quiz-and-questionnaire-api/testing.jsonnet',
            required=True
        )

        eps = sdk2.parameters.String(
            'EPS without DC',
            description='Endpointset without datacenter, e.g. `event-log-api_testing.deploy_unit`',
            default_value='quiz-and-questionnaire-api_testing.deploy_unit',
            required=True
        )

        wait_for_deploy = sdk2.parameters.Bool(
            'Wait for app to deploy',
            description='Wait for application to finish deploy (timeout 1 hour)',
            default=True,
            required=False
        )

        secret_id_for_jsonnet = sdk2.parameters.String(
            'Secret ID for jsonnet',
            description='Secret id for Y.D. application. Task will retreive delegation token for this secret.',
            required=False
        )

        branch = sdk2.parameters.String(
            'Source branch',
            description='Arcadia branch to build and test.'
                        'E.g.: trunk, users/coldmind/somebranch',
            default_value='trunk',
            required=True
        )

        pr_id = sdk2.parameters.String(
            'PR id for building unstable stage',
            description='Pull-request id',
            required=False
        )

        docker_tag = sdk2.parameters.String(
            'Docker tag version',
            description='Image version, e.g. 712345.',
            required=True
        )

        env_secrets = sdk2.parameters.Dict(
            'Secrets for environment',
            default={'sec-01e0cwyhrya60z5qm49pyqxbgk@yav_token': 'YAV_TOKEN',
                     'sec-01e0cwyhrya60z5qm49pyqxbgk@dctl_token': 'DCTL_YP_TOKEN',
                     'sec-01e0cwyhrya60z5qm49pyqxbgk@ya_token': 'YA_TOKEN'},
            description='These secrets are from your robot! E.g. robot-kp-java. This is needed for task.'
                        'Map YAV secret [ID@key] to environment variable name. '
                        '*YAV secret must be delegated to Sandbox*. '
                        'E.g.: sec-xxxx@myval = MYTOKEN',
        )


class JsonnetToDeploy(sdk2.Task):
    name_for_humans = 'Build jsonnet config into YAML, put correct delegation_tokens and deploy it to deploy.'
    Parameters = JsonnetToDeployParameters

    class Requirements(sdk2.Task.Requirements):
        # privileged = True
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC
        cores = 1
        ram = 1  # 1GiB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    @staticmethod
    def _execute_shell(task_instance, cmd, cwd='/tmp/', logger='cmd_log'):
        with ProcessLog(task_instance, logger=logger) as log:
            process = subprocess.Popen(
                cmd,
                shell=True,
                stdout=log.stdout,
                stderr=subprocess.STDOUT,
                cwd=cwd,
            )
            process.wait()
            if process.returncode:
                raise common.errors.TaskFailure('Running command failed, see logs (e.g. cmd_log.log).')

    def _get_jsonnet_binary(self, path_to):
        branch = 'arcadia-arc:/#{}'.format(self.Parameters.branch)
        with arcadia_sdk.mount_arc_path(branch) as path_arc:
            fs_path_ya = path.join(path_arc, 'ya')
            self._execute_shell(self, '{} download sbr:1958114802 --output {}'.format(fs_path_ya, path_to))
            self._execute_shell(self, 'chmod a+x {}'.format(path_to))

    def on_execute(self):
        self._get_jsonnet_binary('/tmp/jsonnet')

        branch = 'arcadia-arc:/#{}'.format(self.Parameters.branch)
        path_to_jsonnet = str(self.Parameters.path_to_jsonnet)

        logging.info('Building config from {} for branch {}'.format(path_to_jsonnet, branch))

        with arcadia_sdk.mount_arc_path(branch) as path_arc:

            fs_path_ya = path.join(path_arc, 'ya')
            fs_path_source = path.join(path_arc, path_to_jsonnet[::-1].split('/', 1)[1][::-1])
            fs_path_to_jsonnet = path.join(path_arc, path_to_jsonnet)

            # Dump secrets from dict into environment variables.
            #
            SecretHelper.initialize_secrets(self.Parameters.env_secrets)

            # Initialize dir structure for jsonnet
            #
            app_and_env = fs_path_to_jsonnet.split('/')[-2:]
            app, env = app_and_env[0], app_and_env[1].split('.')[0]
            self._execute_shell(self, 'mkdir -p /tmp/kinopoisk/deployments/_GENERATED/{app}/{env}/deploy/{{,juggler-checks}} '.format(app=app, env=env) +
                                      '/tmp/kinopoisk/deployments/_GENERATED/{app}/{env}/solomon/{{05-services,10-clusters,20-shards,40-alerts,50-dashboards}}'.format(app=app, env=env))  # noqa: E501

            # Execute building jsonnet configs into json.
            #
            self._execute_shell(self, '/tmp/jsonnet {} --ext-str docker_tag="{}" -m /tmp'.format(
                                fs_path_to_jsonnet,
                                self.Parameters.docker_tag), cwd=fs_path_source)

            # Convert json spec to yaml spec and save it.
            #
            with open('/tmp/kinopoisk/deployments/_GENERATED/{app}/{env}/deploy/deployment.json'.format(app=app, env=env), 'r') as __fd__:
                spec = __fd__.read()
                has_sec = 'sec-' in spec
            yaml.safe_dump(yaml.load(spec), file('/tmp/kinopoisk/deployments/_GENERATED/{app}/{env}/deploy/deployment.yaml'.format(app=app, env=env), 'w'), encoding='utf-8', allow_unicode=True)

            # Put stage
            if has_sec:
                self._execute_shell(self, '{} tool dctl put stage /tmp/kinopoisk/deployments/_GENERATED/{app}/{env}/deploy/deployment.yaml --rewrite-delegation-tokens'.format(fs_path_ya, app=app, env=env))  # noqa: E501
            else:
                self._execute_shell(self, '{} tool dctl put stage /tmp/kinopoisk/deployments/_GENERATED/{app}/{env}/deploy/deployment.yaml'.format(fs_path_ya, app=app, env=env))

            # Here we already know that dctl successfully pushed spec to deploy
            #
            stage_id = self.Parameters.eps.split('.')[0]
            if self.Parameters.wait_for_deploy:
                time.sleep(15)  # Wait for revision to increment
                proc = subprocess.Popen('stdbuf -o0 {} tool dctl get stage {}'.format(fs_path_ya, stage_id), shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                proc.wait()
                spec_str = str(proc.stdout.read())
                target_revision = int(re.findall(r'revision: (\d+)', spec_str)[-1])
                logging.debug('Got target revision from get stage: {}'.format(target_revision))
                # Deploy process: Ready (old rev.) -> InProgress -> Ready (new rev.)
                # First wait for stage to be InProgress. Timeout is 5 min.
                init_deploy_timeout = 5 * 60
                for i in range(init_deploy_timeout):
                    proc = subprocess.Popen('stdbuf -o0 {} tool dctl status stage {}'.format(fs_path_ya, stage_id), shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                    proc.wait()
                    if proc.returncode:
                        logging.error('Error checking status')
                        logging.info('View deploying progress here: https://deploy.yandex-team.ru/stages/{}'.format(stage_id))
                        break
                    status_str = str(proc.stdout.read())
                    logging.info('Current status is:\n' + status_str)
                    revision_from_status = int(re.findall(r'\s+([\d]+)\s+', status_str)[0])
                    logging.debug('Got revision from status stage: {}'.format(revision_from_status))
                    current_status = re.findall(r'Ready|InProgress', status_str)[0]
                    logging.debug('Got current status: {}'.format(current_status))
                    if revision_from_status >= target_revision and current_status in ['InProgress', 'Ready']:
                        break
                    if i == init_deploy_timeout - 1:
                        if revision_from_status >= target_revision and current_status in ['InProgress', 'Ready']:
                            break
                        else:
                            logging.error('Reached timeout 5 min in starting deploying')
                            raise common.errors.TaskFailure('Stage is not deploying https://deploy.yandex-team.ru/stages/{}'.format(stage_id))
                    time.sleep(1)
                logging.info('Stage started deploying. Now wait for it to finish deploying.')
                # Now wait for stage to be Ready. Timeout is 60 min.
                finish_deploy_timeout = 60 * 60
                for i in range(finish_deploy_timeout):
                    proc = subprocess.Popen('stdbuf -o0 {} tool dctl status stage {}'.format(fs_path_ya, stage_id), shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                    proc.wait()
                    if proc.returncode:
                        logging.error('Error checking status')
                        logging.info('View deploying progress here: https://deploy.yandex-team.ru/stages/{}'.format(stage_id))
                        break
                    status_str = str(proc.stdout.read())
                    logging.info('Current status is:\n' + status_str)
                    current_status = re.findall(r'Ready|InProgress', status_str)[0].strip()
                    logging.debug('Got current status: {}'.format(current_status))
                    logging.info('+' + current_status + '+')
                    logging.info('Checking if current status is `Ready`...')
                    if current_status in ['Ready']:
                        logging.info('Successfully deployed!!! https://deploy.yandex-team.ru/stages/{}'.format(stage_id))
                        break
                    if i == finish_deploy_timeout - 1:
                        if current_status in ['Ready']:
                            logging.info('Successfully deployed!!! https://deploy.yandex-team.ru/stages/{}'.format(stage_id))
                            break
                        else:
                            logging.error('Reached timeout 1 hour in deploying')
                            raise common.errors.TaskFailure('Stage is not deploying :( https://deploy.yandex-team.ru/stages/{}'.format(stage_id))
                    time.sleep(1)
            else:
                logging.info('View deploying progress here: https://deploy.yandex-team.ru/stages/{}'.format(stage_id))


# EOF
