# -*- coding: utf-8 -*-

"""
    DESCRIPTION: Sandbox task for managing unstables in awacs, e.g. creating
                 upstreams, backends, updating domains, etc.
                 Also deleting all this (not domains).
    AUTHOR: @coldmind
    ST_QUEUE: CADMIN
"""


import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

import logging
import json

from os import path

from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.media.utils.secret_helper import SecretHelper


class ConfigException(Exception):
    pass


class ManageUnstablesInAwacsParameters(sdk2.Parameters):
    container = sdk2.parameters.Container('Container', required=True,
                                          platform='linux_ubuntu_18.04_bionic')

    with sdk2.parameters.Group('Manage unstables in awacs task parameters') as group_parameters:
        megaconfig = sdk2.parameters.JSON(
            'Config for script',
            default={
                'action': 'insert_unstable',
                'endpoint_set': 'quiz-and-questionnaire-api_unstable.pr-5678',
                'namespace': 'kp-nonprod-balancer.kp.yandex.net',
                'datacenters': ['man'],
                'fqdns': ['pr-5678-quiz-and-questionnaire-api.unstable.kp.yandex.net'],
                'domain': 'unstable.kp.yandex.net',
                'guids': ['61458', '31277'],
                'users': ['robot-kp-java'],
                'wait': 'False'
            },
            description='JSON config used in script for unstables. '
                        'Info on every parameter: https://wiki.yandex-team.ru/cult-admin/tulza-dlja-zavedenija-anstejjblov-v-avakse/'
        )

        branch = sdk2.parameters.String(
            'Source branch',
            description='Arcadia branch to build and test.'
                        'E.g.: trunk, users/coldmind/somebranch',
            default_value='trunk',
            required=True
        )

        path_to_awacs_tool = sdk2.parameters.String(
            'Arcadia path to target script',
            description='Path to directory with `main.py` of awacs_stage_inserter',
            default_value='kinopoisk/sre/deploy/awacs',
            required=False
        )

        build_fresh_awacs_tool = sdk2.parameters.Bool(
            'Build fresh awacs stage inserter (LONG)',
            description='If this True, build (YA_MAKE) the fresh release of awacs tool'
                        ' from branch in parameters with path from parameters.'
                        ' This step will take 20 min!'
                        ' If False, do not build and use latest resource AWACS_STAGE_INSERTER',
            default=False,
            required=False
        )

        env_secrets = sdk2.parameters.Dict(
            'Secrets for environment',
            default={'sec-01e0cwyhrya60z5qm49pyqxbgk@awacs_token': 'AWACS_TOKEN'},
            description='These secrets are from your robot! E.g. robot-kp-java. This is needed for task.'
                        'Map YAV secret [ID@key] to environment variable name. '
                        '*YAV secret must be delegated to Sandbox*. '
                        'E.g.: sec-xxxx@myval = MYTOKEN',
        )


class ManageUnstablesInAwacs(sdk2.Task):
    name_for_humans = 'Insert unstable or remove unstable, pointing to specifi YP EPS in awacs'
    Parameters = ManageUnstablesInAwacsParameters

    class Requirements(sdk2.Task.Requirements):
        # privileged = True
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC
        cores = 1
        ram = 1  # 1GiB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def _build(self, target, output_directory):
        branch = 'arcadia-arc:/#{}'.format(self.Parameters.branch)
        with arcadia_sdk.mount_arc_path(branch) as path_arc:
            arcadia_sdk.do_build(
                build_system=sdk_constants.DISTBUILD_BUILD_SYSTEM,
                source_root=path_arc,
                targets=[target],
                results_dir=output_directory,
                patch=None,
                clear_build=False
            )

    def _validate_config(self):
        logging.info('Validate config.')
        try:
            _ = dict(self.Parameters.megaconfig)
        except ValueError:
            raise ConfigException('Invalid config. Please check import config(JSON)')

    def on_prepare(self):
        # Validate config.
        #
        self._validate_config()

    def on_execute(self):
        # Put config into file.
        #
        config_path = '/tmp/config.json'
        with open(config_path, 'w') as __fd__:
            json.dump(self.Parameters.megaconfig, __fd__, ensure_ascii=False, indent=4)

        # Dump secrets from dict into environment variables.
        #
        SecretHelper.initialize_secrets(self.Parameters.env_secrets)

        # If need to build fresh release of tool, do it.
        #
        if self.Parameters.build_fresh_awacs_tool:
            # Build awacs stage inserter tool.
            #
            self._build(self.Parameters.path_to_awacs_tool, '/tmp/')
            # Full path should be something like this `/tmp/kinopoisk/sre/deploy/awacs/awacs_stage_inserter`
            #
            self.Context.path_to_awacs_stage_inserter = path.join('/tmp', self.Parameters.path_to_awacs_tool, 'awacs_stage_inserter', 'awacs_stage_inserter')
            self.Context.save()
        else:
            # Else we take latest released resource of needed type.
            #
            resource_type = 'AWACS_STAGE_INSERTER'
            logging.info('Prepare resource {}'.format(resource_type))
            resource = sdk2.Resource[resource_type].find()
            if resource.count > 0:
                res = resource.first()
                logging.debug('Got first resource of type {}'.format(resource_type))
                logging.debug(str(res.__dict__))
                binary = sdk2.ResourceData(res)
                self.Context.path_to_awacs_stage_inserter = path.join(str(binary.path), 'awacs_stage_inserter', 'awacs_stage_inserter')
                self.Context.save()
            else:
                logging.error('{} not found. Try to build it(copy from https://sandbox.yandex-team.ru/task/901138959/)'.format(resource_type))

        logging.debug('Path to awacs_stage_inserter is {}'.format(self.Context.path_to_awacs_stage_inserter))

        # Do work.
        #
        run_process(['chmod', 'a+x', self.Context.path_to_awacs_stage_inserter], log_prefix='chmod_for_binarnik')
        run_process([self.Context.path_to_awacs_stage_inserter, '--config', config_path], log_prefix='awacs_stage_inserter')


# EOF
