# -*- coding: utf-8 -*-
"""
    DESCRIPTION: Sandbox task for build nginx config for kp-front nanny service
    AUTHOR: @coldmind
    ST_QUEUE: CADMIN
"""

from __future__ import unicode_literals

import os
import sys
import tarfile
import logging

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.media.resources import KpFrontNginxBinary


class KpFrontNginxBinaryParameters(sdk2.Parameters):
    with sdk2.parameters.Group('KpFront nginx binary build parameters') as kp_front_binary:
        arc_url = sdk2.parameters.String(
            'Arc nginx repository url',
            default_value='svn+ssh://coldmind@arcadia.yandex.ru/arc/trunk/arcadia/nginx/bin-noperl',
            required=True
        )


class MakeKpFrontNginxBinary(sdk2.Task):
    name_for_humans = 'Build kp front nginx binary'

    Parameters = KpFrontNginxBinaryParameters

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC

    def execute_with_logging(self, cmd):
        logging.debug('Running {}'.format(cmd))
        s = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE, shell=True)
        exitcode = None
        while True:
            line = s.stderr.readline()
            exitcode = s.poll()

            if (not line) and (exitcode is not None):
                break
            logging.debug('pid %s: %s', os.getpid(), line.strip())

        logging.debug('pid %s: finished', os.getpid(), cmd)
        if exitcode != 0:
            sys.exit(-1)
        return exitcode

    def on_execute(self):
        self.svn_repo()
        self.create_resource()

    def svn_repo(self):
        self.Context.arcadia_root_path = os.path.abspath('arcadia')
        self.Context.nginx_bin_dir = os.path.abspath('arcadia/nginx/bin-noperl')

        logging.info('-> Checking out basic arcadia')
        self.execute_with_logging('svn co --depth files --non-interactive\
                svn+ssh://coldmind@arcadia.yandex.ru/arc/trunk/arcadia\
                {}'.format(self.Context.arcadia_root_path))

        logging.info('-> Checking out yandex nginx')
        self.execute_with_logging('cd {} && svn co --depth files\
                --non-interactive {}'.format(self.Context.arcadia_root_path, self.Parameters.arc_url))

        logging.info('-> YA MAKE nginx binary and checkout all deps (time-consuming, downloads ~420Mb)')
        self.execute_with_logging('cd {} && ya make --checkout'.format(self.Context.nginx_bin_dir))

    def create_resource(self, path=None, filename='kp_front_nginx_bin.tar'):
        """
            Create tar archive with bin and publish it
        """
        _resource_ = os.path.abspath(filename)

        logging.info('-> Compressing nginx binary directory')
        with tarfile.TarFile(_resource_, 'w') as _tar_:
            _tar_.add(os.path.join(self.Context.nginx_bin_dir, 'nginx'), 'bin')

        _desc_ = '{} build from {}'.format(
            self.name_for_humans,
            self.Parameters.arc_url
        )

        logging.info('-> Creating resource: {}'.format(filename))
        KpFrontNginxBinary(self, _desc_, _resource_)

# EOF
