# -*- coding: utf-8 -*-
"""
    DESCRIPTION: Sandbox task for build nginx config for kp-front nanny service
    AUTHOR: @coldmind
    ST_QUEUE: CADMIN
"""

from __future__ import unicode_literals

import os
import sys
import re
import logging
import tarfile
import inspect

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.common.errors import TaskFailure

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.nanny import nanny


class ValidateKpFrontNginxConfigsParameters(sdk2.Parameters):
    container = sdk2.parameters.Container('Container', required=True,
                                          platform='linux_ubuntu_16.04_xenial')

    with sdk2.parameters.Group('KpFront nginx validator parameters') as kp_front_bundle:
        arc_url = sdk2.parameters.String(
            'Arc repository url',
            default_value='svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/kinopoisk/sre/salt/kp-front',
            required=True
        )
        salt_grains = sdk2.parameters.String(
            'Salt minion grains',
            multiline=True,
            required=True
        )


class ValidateKpFrontNginxConfigs(nanny.ReleaseToNannyTask2, sdk2.Task):
    name_for_humans = 'Build configs by envs'

    Parameters = ValidateKpFrontNginxConfigsParameters

    class Requirements(sdk2.Task.Requirements):
        privileged = True
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_XENIAL

    def execute_with_logging(self, cmd):
        logging.debug('Running {}'.format(cmd))
        s = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE, shell=True)
        exitcode = None
        while True:
            line = s.stderr.readline()
            exitcode = s.poll()

            if (not line) and (exitcode is not None):
                break
            logging.debug('pid %s: %s', os.getpid(), line.strip())

        logging.debug('pid %s: finished', os.getpid(), cmd)
        # if exitcode != 0:
        #     sys.exit(-1)
        return exitcode

    def on_execute(self):
        self.svn_repo()
        self.git_repo()
        self.svn_frontend_repo()

        self.fetch_oauth()

        self.install_salt()
        self.set_grains()
        self.prerequisites()
        self.fix_certifi()
        self.make_configs()

        self.create_resources()

    def svn_repo(self):
        self.Context.nginx_salt_configs_path = os.path.abspath('/tmp/arcadia/kinopoisk/apps/kp-front')
        self.execute_with_logging('mkdir -p {}'.format(self.Context.nginx_salt_configs_path))
        logging.info('-> Checking out nginx salt configs from arcadia')
        self.execute_with_logging('svn co {} {} --non-interactive'.format(self.Parameters.arc_url, self.Context.nginx_salt_configs_path))
        return self.Context.nginx_salt_configs_path

    def git_repo(self):
        """
        Gets templates from git repo "common"
        """
        git_repo_dir = '/tmp/git_repo'
        run_process(['apt', 'install', 'git'])
        run_process(['git', 'clone', 'https://github.yandex-team.ru/salt-media/common.git', git_repo_dir])
        run_process(['cp', '-r', '{}/roots/templates/nginx'.format(git_repo_dir), '{}/roots/templates'.format(self.Context.nginx_salt_configs_path)])
        run_process(['cp', '-r', '{}/roots/templates/loggiver'.format(git_repo_dir), '{}/roots/templates'.format(self.Context.nginx_salt_configs_path)])
        run_process(['cp', '-r', '{}/roots/templates/push-client'.format(git_repo_dir), '{}/roots/templates'.format(self.Context.nginx_salt_configs_path)])
        run_process(['cp', '-r', '{}/roots/templates/certificates'.format(git_repo_dir), '{}/roots/templates'.format(self.Context.nginx_salt_configs_path)])

    def svn_frontend_repo(self):
        self.arc_frontend_url = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/kinopoisk/frontend/kp/packages/kp-stub/src'
        self.kp_stub_path = os.path.abspath('/tmp/kp-stub')
        self.execute_with_logging('mkdir -p {}'.format(self.kp_stub_path))
        logging.info('-> Checking out frontend kp-stub configs from arcadia')
        self.execute_with_logging('svn co {} {} --non-interactive'.format(self.arc_frontend_url, self.kp_stub_path))
        self.Context.kp_stub_path = self.kp_stub_path

    def fetch_oauth(self):
        try:
            self.oauth_testing = sdk2.Vault.data('sec-01crx9aa5tyrs9xgpwh0thbz75[robot-kp-yav-testing]')
            self.oauth_stable = sdk2.Vault.data('sec-01crx9aa5tyrs9xgpwh0thbz75[robot-kp-yav-stable]')
        except:
            raise TaskFailure('Error fetching vault data.')

    def write_oauth(self, secret):
        with open('/etc/salt/minion.d/yav.conf', 'w') as __fd__:
            __fd__.write('yav.config:\n  oauth-token: {}\n'.format(secret))

    def install_salt(self):
        run_process(['apt', 'update'])
        run_process(['apt',  '--yes',  '--force-yes', 'install', 'salt-minion=2018.3.4-yandex3', 'salt-common=2018.3.4-yandex3', 'salt-yandex-components=2018.3.4-yandex3'])
        # And install passport and paramiko deps
        #
        run_process(['apt',  '--yes',  '--force-yes', 'install', 'yandex-passport-vault-client', 'python-paramiko'])

    def fix_certifi(self):
        run_process(['mkdir', '/tmp/certifi'])
        run_process(['wget', 'https://proxy.sandbox.yandex-team.ru/3122550795', '-O', '/tmp/certifi/certifi.zip'])
        run_process(['unzip', '/tmp/certifi/certifi.zip', '-d', '/tmp/certifi'])
        run_process(['rm', '-Rf', '/usr/local/lib/python2.7/dist-packages/certifi'])
        run_process(['cp', '-R', '/tmp/certifi/python-certifi-2021.10.08/certifi', '/usr/local/lib/python2.7/dist-packages/'])

    def set_grains(self):
        grains_path = '/etc/salt'
        import yaml
        try:
            self.Context.envs = yaml.load(self.Parameters.salt_grains)
        except yaml.YAMLError:
            raise yaml.YAMLError('Failed load yaml salt grains')

        if not os.path.exists(grains_path):
            logging.info('-> Directory `{}` does not exist; creating...'.format(grains_path))
            self.execute_with_logging('mkdir -p {}'.format(grains_path))
            self.execute_with_logging('touch /etc/salt/grains')

        grains_path = '/etc/salt/grains'

        with open('/tmp/grains', 'w') as f:
            f.write(self.Parameters.salt_grains)
        run_process(['cp', '/tmp/grains', grains_path])

    def prerequisites(self):
        run_process(['apt', 'install', '--yes', 'python-pip'])
        run_process(['python', '-m', 'pip', 'install', 'requests'])
        run_process(['apt', 'install', '--yes', 'nginx=1.14.2-1.yandex.6'])
        run_process(['apt', 'purge', '--yes', 'yandex-push-client'])
        run_process(['apt', 'download', '--yes', 'yandex-push-client=6.33.0'])
        run_process(['dpkg', '--unpack', 'yandex-push-client_6.33.0_amd64.deb'])
        self.execute_with_logging('echo "/bin/true" > /var/lib/dpkg/info/yandex-push-client.postinst')
        run_process(['dpkg', '--configure', '-D', '777', 'yandex-push-client'])
        run_process(['groupadd', 'statbox'])
        run_process(['useradd', 'statbox', '-g', 'statbox'])
        run_process(['cp', '/etc/nginx/mime.types', '/tmp/'])
        run_process(['rm', '-Rf', '/etc/nginx/*'])
        run_process(['cp', '/tmp/mime.types', '/etc/nginx/'])
        run_process(['apt', 'install', '--yes', 'luarocks'])
        run_process(['luarocks', 'install', 'lua-protobuf'])

    def make_configs(self):
        _envs_ = list(self.Context.envs['envs'])
        for e in _envs_:
            configs_dir_by_env = '/tmp/configs_{}/'.format(e)
            run_process(['mkdir', configs_dir_by_env])
            self.execute_with_logging('rm -Rf /etc/nginx')
            self.execute_with_logging('mkdir /etc/nginx')
            self.execute_with_logging('cp /tmp/mime.types /etc/nginx/')
            process = sp.Popen(['ls', '-lath', '/etc/nginx'], stdout=sp.PIPE, stderr=sp.STDOUT)
            stdout = process.communicate()[0]
            logging.debug('Contents of /etc/nginx/ BEFORE salt-calling with env {} (sould be only mime.types!):\n\n{}\n\n'.format(e, stdout))
            self.ch_env(e)
            if e in ['production', 'prestable']:
                self.write_oauth(self.oauth_stable)
            else:
                self.write_oauth(self.oauth_testing)

            with ProcessLog(self, logger='salt_call_front_{}'.format(e)) as pl:
                output = sp.check_output(
                    ['salt-call', 'state.sls', 'front', '-l', 'debug',
                     '--file-root',
                     '{}/roots'.format(self.Context.nginx_salt_configs_path),
                     '--pillar-root',
                     '{}/pillar'.format(self.Context.nginx_salt_configs_path),
                     '--local'],
                    stderr=pl.stderr
                )
                logging.info('salt-call state.sls front output is: \n\n\n\n {}'.format(str(output)))

            with ProcessLog(self, logger='salt_call_push_client_{}'.format(e)) as pl:
                output = sp.check_output(
                    ['salt-call', 'state.sls', 'templates.push-client', '-l',
                     'debug', '--file-root',
                     '{}/roots'.format(self.Context.nginx_salt_configs_path),
                     '--pillar-root',
                     '{}/pillar'.format(self.Context.nginx_salt_configs_path),
                     '--local'],
                    stderr=pl.stderr
                )
                logging.info('salt-call state.sls push_client output is: \n\n\n\n {}'.format(str(output)))

            self.make_env_link(e)

            # Test generated config.
            #
            self.test_config()

            # IF configs are valid then copy them to tmp dir for furher resource creation.
            # Copy all necessary files to temp dir.
            #
            self.execute_with_logging('mv /etc/nginx {}'.format(configs_dir_by_env))
            self.execute_with_logging('mkdir /etc/nginx')
            process = sp.Popen(['ls', '-lath', '/etc/nginx'], stdout=sp.PIPE, stderr=sp.STDOUT)
            stdout = process.communicate()[0]
            logging.debug('Contents of /etc/nginx/ AFTER salt-calling with env {} (sould be empty!):\n\n{}\n\n'.format(e, stdout))
            self.execute_with_logging('cp /etc/logrotate.d/logrotate-nginx {}'.format(configs_dir_by_env))
            self.execute_with_logging('cp /usr/bin/strip-password-in-url.pl {}'.format(configs_dir_by_env))
            self.execute_with_logging('cp {}/roots/front/files/loggiver.pattern {}'.format(self.Context.nginx_salt_configs_path, configs_dir_by_env))
            self.execute_with_logging('cp /etc/default/push-client {}/push-client-{}'.format(configs_dir_by_env, e))
            self.execute_with_logging('cp -R /etc/yandex/statbox-push-client {}'.format(configs_dir_by_env))
            self.execute_with_logging('cp -R {} {}'.format(self.kp_stub_path, configs_dir_by_env))  # Copy kp-stub

    def make_env_link(self, env_name):
        self.execute_with_logging('cd /etc/nginx && ln -sfT {}/environments environments'.format(env_name))

    def ch_env(self, env_name):
        with open('/etc/yandex/environment.type', 'w') as __fd__:
            __fd__.write(env_name)
        lines = ''
        with open('/etc/yandex/environment.type.xml', 'r') as __fd__:
            lines = ''.join(__fd__.readlines())
        regexp = '<environment>{}</environment>'
        res = re.sub(regexp.format('(\w+)'), regexp.format(env_name), lines)
        with open('/etc/yandex/environment.type.xml', 'w') as __fd__:
            for line in res.split('\n'):
                __fd__.write(line)

    def test_config(self):
        """
            Tests config syntax for one (current) env
        """
        ret_code = self.execute_with_logging('nginx -t 2>&1 | grep "syntax is ok"')
        if ret_code != 0:
            process = sp.Popen(['nginx', '-t'], stdout=sp.PIPE, stderr=sp.STDOUT)
            stdout = process.communicate()[0]
            logging.debug('Testing NGINX config:\n\n{}\n\n'.format(stdout))
            raise RuntimeError('Exception when running ```{}``` command!\
            Either NGINX config is not valid, or problem (mode, permissions, existence) with files!'.format('nginx -t'))

    def create_resources(self, path=None):
        """
            Create tar archive with bundle and publish it
        """
        _envs_ = list(self.Context.envs['envs'])
        for e in _envs_:
            filename = 'kp-front-{}.tar'.format(e)
            configs_dir = '/tmp/configs/'
            configs_dir_by_env = '/tmp/configs_{}'.format(e)

            if os.path.isdir(configs_dir):
                self.execute_with_logging('rm -Rf {}/*'.format(configs_dir))
            else:
                self.execute_with_logging('mkdir {}'.format(configs_dir))

            self.execute_with_logging('cp -R {}/* {}'.format(configs_dir_by_env, configs_dir))

            _resource_ = '{}/../{}'.format(configs_dir, filename)
            resources = inspect.getmembers(sys.modules['sandbox.projects.media.resources'], inspect.isclass)
            for r in resources:
                if r[0].lower().startswith('kpfrontconfigs') and e in r[0].lower():
                    res = getattr(sys.modules['sandbox.projects.media.resources'], r[0])

            logging.info('-> Compressing salt nginx configs  {}  directory'.format(e))
            with tarfile.TarFile(_resource_, 'w') as _tar_:
                _tar_.add('{}'.format(configs_dir), 'configs')

            _desc_ = '{} build from {}'.format(
                self.name_for_humans,
                self.Parameters.arc_url
            )

            self.execute_with_logging('cp {} {}'.format(_resource_, os.path.abspath('.')))

            logging.info('-> Creating resource: {}'.format(filename))
            res(self, _desc_, os.path.join(os.path.abspath('.'), filename))

# EOF
