import logging
from os import environ
from collections import namedtuple
from sandbox.sdk2 import yav


class SecretHelper:
    @staticmethod
    def initialize_secrets(mapping):
        '''
        Get secret data from YAV by mapping and put it into environment vars.

        :param mapping: sec-id@key = VARNAME

        '''
        by_secret = {}
        SecretMeta = namedtuple('SecretMeta', ['alias', 'entries'])
        SecretEntry = namedtuple('SecretEntry', ['key', 'alias'])

        for name_secret, name_env in mapping.items():
            secret_id, _, secret_key = name_secret.partition('@')
            name_env = (name_env or secret_key).upper()

            logging.debug('Secret mapping: id {}, key {} -> {}'.format(secret_id, secret_key, name_env))

            secret_tuple = SecretEntry(secret_key, name_env)

            if secret_id in by_secret:
                by_secret[secret_id].entries.append(secret_tuple)

            else:
                by_secret[secret_id] = SecretMeta(
                    alias='sec{}'.format(str(len(by_secret) + 1)),
                    entries=[secret_tuple]
                )

        call_kwargs = {
            secret_meta.alias: yav.Secret(secret_id)
            for secret_id, secret_meta in by_secret.items()}

        if not call_kwargs:
            return

        secrets = yav.Yav(**call_kwargs)

        for secret_id, secret_meta in by_secret.items():
            secret_data = getattr(secrets, secret_meta.alias, None)

            if secret_data is None:
                logging.warning('No secret data from YAV for {}'.format(secret_id))
                continue

            for (name_key, name_env) in secret_meta.entries:
                value = secret_data.get(name_key, None)

                if value is None:
                    logging.warning('No secret data from YAV for {}@{}'.format(secret_id, name_key))
                    continue

                environ[name_env] = value
