# coding: utf-8

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.media_crm.resource_types import MediaCrmBinary


BINARY_TASK_ATTR_TARGET = 'media_crm/tasks/media_crm_build_docker/bin'

ARC_SVN_URL = 'arcadia:/arc/trunk/arcadia'

DOCKER_REGISTRY = 'registry.yandex.net'
DOCKER_REGISTRY_PREFIX = 'media-crm'
DOCKER_USER = 'robot-media-crm'
DOCKER_TOKEN_VAULT_NAME = '{}.docker-token'.format(DOCKER_USER)


class MediaCrmBuildDocker(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Settings') as settings_block:
            package_path = sdk2.parameters.String(
                'Package path (relative to arcadia)',
                required=True)
            arc_svn_url = sdk2.parameters.String(
                'Svn url for arc with revision',
                default=ARC_SVN_URL,
                required=True)

        with sdk2.parameters.Group('Build settings') as build_settings_block:
            UseLastBinary = sdk2.parameters.Bool('Use last binary archive',
                                                 default=True)
            with UseLastBinary.value[True]:
                with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                    ReleaseType.values.stable = ReleaseType.Value('stable',
                                                                  default=True)
                    ReleaseType.values.test = ReleaseType.Value('test')
            with UseLastBinary.value[False]:
                custom_tasks_archive_resource = sdk2.parameters.Resource(
                    'task archive resource',
                    default=None, )

    def on_save(self):
        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={'target': BINARY_TASK_ATTR_TARGET,
                       'release': self.Parameters.ReleaseType or 'stable'}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def _build_docker(self):
        build_docker_task = sdk2.Task['YA_PACKAGE'](
            self,
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            description='Building and publishing Docker image',

            # Arcadia
            checkout_arcadia_from_url=self.Parameters.arc_svn_url,
            checkout=False,

            # Packages
            packages=self.Parameters.package_path,
            package_type='docker',
            resource_type=MediaCrmBinary.name,

            docker_registry=DOCKER_REGISTRY,
            docker_image_repository=DOCKER_REGISTRY_PREFIX,
            docker_user=DOCKER_USER,
            docker_token_vault_name=DOCKER_TOKEN_VAULT_NAME,
            docker_push_image=True,
            docker_build_network='host',

            # Advanced, YT store params
            ya_yt_store=False,
            use_aapi_fuse=True,
            ignore_recurses=True
        )

        build_docker_task.enqueue()
        self.Context.ya_package_task_id = build_docker_task.id
        raise sdk2.WaitTask(
            [build_docker_task],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
        )

    def on_execute(self):
        with self.memoize_stage.build_docker:
            self._build_docker()
        self.Context.result_registry_url = sdk2.Task[self.Context.ya_package_task_id].Context.output_resource_version
