# coding: utf-8

import logging
import re

import sandbox.sdk2 as sdk2
from sandbox import common
import sandbox.common.types.task as ctt
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.media_crm.resource_types import MediaCrmBinary


ARC_SVN_URL = 'arcadia:/arc/trunk/arcadia'

BINARY_TASK_ATTR_TARGET = 'media_crm/tasks/media_crm_deploy/bin'

DEFAULT_ST_USER = 'robot-media-crm'
DEFAULT_QLOUD_USER = DEFAULT_ST_USER
QLOUD_COMPONENTS_MODES = {
    'media-crm.crm-api': {
        'testing': ['testing'],
        'production': ['production']
    },
    'media-crm.crm-worker': {
        'testing': ['testing'],
        'production': ['production']
    },
    'media-crm.crm-lbm': {
        'testing': ['testing'],
        'production': ['production']
    },
    'media-crm.crm-cms': {
        'testing': ['testing'],
        'production': ['production']
    }
}

REGISTRY_URL = 'registry.yandex.net/media-crm/media-crm'
ST_AGILE_BOARD_ID = 3260


class MediaCrmDeploy(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.RadioGroup("Deployment mode") as mode:
            mode.values['testing'] = mode.Value('Testing workflow')
            mode.values['production'] = mode.Value('Production workflow', default=True)

        with mode.value['production']:
            st = sdk2.parameters.String('Startrek release ticket url', required=True)

        with mode.value['testing']:
            package_path = sdk2.parameters.String('Package path (relative to arcadia)', required=True)
            arc_svn_url = sdk2.parameters.String('Svn url for arc with revision', default=ARC_SVN_URL, required=True)

        with sdk2.parameters.CheckGroup("Application", required=True) as applications:
            for name in QLOUD_COMPONENTS_MODES:
                setattr(applications.values, name, applications.Value(name, checked=True))

        with sdk2.parameters.Group('Build settings') as build_settings_block:
            UseLastBinary = sdk2.parameters.Bool('Use last binary archive',
                                                 default=True)
            with UseLastBinary.value[True]:
                with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                    ReleaseType.values.stable = ReleaseType.Value('stable',
                                                                  default=True)
                    ReleaseType.values.test = ReleaseType.Value('test')
            with UseLastBinary.value[False]:
                custom_tasks_archive_resource = sdk2.parameters.Resource(
                    'task archive resource',
                    default=None, )

    def on_save(self):
        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={'target': BINARY_TASK_ATTR_TARGET,
                       'release': self.Parameters.ReleaseType or 'stable'}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def _build_docker(self):
        build_docker_task = sdk2.Task['MEDIA_CRM_BUILD_DOCKER'](
            self,
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            description='Building and publishing Docker image',
            arc_svn_url=self.Parameters.arc_svn_url,
            package_path=self.Parameters.package_path,
        )

        build_docker_task.enqueue()
        self.Context.build_docker_task_id = build_docker_task.id
        raise sdk2.WaitTask(
            [build_docker_task],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
        )

    def _deploy(self):
        deploy_with_robot = True
        if self.Parameters.mode == 'testing':
            self.Context.registry_url = sdk2.Task[
                self.Context.build_docker_task_id].Context.result_registry_url
            self.Context.target_state = 'DEPLOYED'
        else:
            if self.__st_release_task.status.key != 'closed':
                raise common.errors.TaskError(
                    "%s task not in tested state (%s)" %
                    (self.Context.release_st_task,
                     self.__st_release_task.status.key))
            for comment in self.__st_release_task.comments.get_all():
                if u'отправлен на выкатку в testing' in comment.text:
                    self.Context.build_task = re.search('Build task: \(\([^\s]+\s(\d+)\)\)', comment.text).group(1)
            logging.info('Build task from testing: {}'.format(self.Context.build_task))
            build_task_ctx = sdk2.Task[self.Context.build_task].Context
            self.Context.registry_url = build_task_ctx.registry_url
            self.Context.target_state = 'COMMITTED'
            deploy_with_robot = False

        result_tasks = []
        for app in self.Parameters.applications:
            for app_env in QLOUD_COMPONENTS_MODES[app].get(self.Parameters.mode, []):
                result_app = '{}.{}'.format(app, app_env)
                deploy_task = sdk2.Task['AFISHA_DEPLOY'](
                    self,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description='Deploying {}'.format(result_app),
                    qloud_environments=result_app,
                    registry_url=self.Context.registry_url,
                    tgt_state=self.Context.target_state,
                    comment=self.Context.release_st_task,
                    deploy_with_robot=deploy_with_robot,
                    deployer_username=DEFAULT_QLOUD_USER
                )
                deploy_task.enqueue()
                result_tasks.append(deploy_task)
        self.Context.deploy_tasks = [task.id for task in result_tasks]
        raise sdk2.WaitTask(
            result_tasks,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
        )

    def __get_release_num(self):
        r = 'arcadia:/arc/(?:branches|tags)/media-crm/media-crm/stable-(\d+)-\d+.*'
        try:
            return re.search(r, self.Parameters.arc_svn_url).group(1)
        except AttributeError:
            return 0

    def on_execute(self):
        import startrek_client
        if self.Parameters.mode == 'production':
            self.__st = startrek_client.Startrek(token=sdk2.Vault.data(self.author, '{}.st-token'.format(self.author)), useragent=DEFAULT_ST_USER)
            self.__st_release_task = self.__st.issues[self.Parameters.st.split('/')[-1]]
            for tag in self.__st_release_task.tags:
                if tag.startswith('release_number_'):
                    self.Context.release_num = tag[len('release_number_'):]
                    break
        else:
            self.__st = startrek_client.Startrek(token=sdk2.Vault.data(DEFAULT_ST_USER, '{}.st-token'.format(DEFAULT_ST_USER)), useragent=DEFAULT_ST_USER)
            self.Context.release_num = self.__get_release_num()
            logging.info('Release number: {}'.format(self.Context.release_num))
            self.__st_release_task = self.__st.issues.find("Queue: ALETCRM AND Tags: release_number_{}".format(self.Context.release_num))[0]

        if len(self.__st_release_task.sprint) == 0:
            board = self.__st.boards[ST_AGILE_BOARD_ID]
            sprints = list(filter(lambda x: x.name.startswith("crm-release " + str(self.Context.release_num)), board.sprints))
            logging.info('Found sprints: {}'.format(sprints))
            if (len(sprints) == 1):
                self.__st_release_task.update(sprint=sprints)

        self.Context.release_st_task = self.__st_release_task.key

        if self.Parameters.mode == 'testing':
            with self.memoize_stage.build:
                self._build_docker()

            resource = sdk2.ResourceData(
                MediaCrmBinary(self, "Output file", "filename.txt"))
            resource.path.write_bytes("Mock for RM!")
            resource.ready()

        with self.memoize_stage.deploy:
            self._deploy()

        deployed_envs = []
        for task_id in self.Context.deploy_tasks:
            for env in sdk2.Task[task_id].Context.result_envs:
                deployed_envs.append(env["url"])
        self.Context.deployed_envs = deployed_envs

        if self.Parameters.mode == 'testing':
            logging.info('Startrek transitions: {}'.format(list(self.__st_release_task.transitions)))
            if self.__st_release_task.status.key == 'closed':
                self.__st_release_task.transitions['open'].execute()
        else:
            docker_tag = self.Context.registry_url.split(':')[1]
            self.__st_release_task.update(assignee=self.author)
            self.__st_release_task.comments.create(
                text='Committed {} release to envs, need manual deploy from mantainers:\n{}'.format(
                    docker_tag,
                    ''.join(['  * {}\n'.format(x) for x in self.Context.deployed_envs])))
            self.server.release(
                task_id=self.Context.build_task,
                type=ctt.ReleaseStatus.STABLE,
                subject=self.Context.release_st_task
            )
