import logging
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.release_machine.helpers import arcanum_helper


ARC_SVN_URL = "arcadia:/arc/trunk/arcadia"
DEFAULT_SUBTASK_WAIT_STATUS = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK


class AfishaArcanumApi(arcanum_helper.ArcanumApi):
    def get_review_request(self, rr_id, **kwargs):
        return self._do_get("v1/review-requests/{}".format(rr_id), params=kwargs)


class MaSandboxBaseTask(sdk2.Task):
    """ Base mediaanalyst sandbox task """

    BINARY_TASK_ATTR_TARGET = None

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Task resource settings", collapse=True) as build_settings_block:
            UseLastBinary = sdk2.parameters.Bool("Use last binary archive", default=True)
            with UseLastBinary.value[True]:
                with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                    ReleaseType.values.stable = ReleaseType.Value("stable", default=True)
                    ReleaseType.values.test = ReleaseType.Value("test")
            with UseLastBinary.value[False]:
                custom_tasks_archive_resource = sdk2.parameters.Resource("task archive resource", default=None)

    def on_save(self):
        if self.BINARY_TASK_ATTR_TARGET is None:
            raise RuntimeError("MediaAnalystSandboxBaseTask must define BINARY_TASK_ATTR_TARGET attribute")

        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={"target": self.BINARY_TASK_ATTR_TARGET,
                       "release": self.Parameters.ReleaseType or "stable"}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def _init_staff(self, user):
        """
        provides self.staff object with staff api client
        :return: None
        """
        from afisha.infra.libs.staff import StaffApiClient
        vault = "{}.st-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.staff = StaffApiClient(token=token)

    def _init_st(self, user, useragent="MediaanalystSandbox"):
        """
            Provides self.st object with startrek client
        """
        import startrek_client
        vault = "{}.st-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.st = startrek_client.Startrek(token=token, useragent=useragent)

    def _init_dc(self, user):
        """
        Provides self.dc object with deployer client
        :param user: owner of secret
        :return: None
        """
        from afisha.infra.libs.deployer import DeployerClient
        vault = "{}.qloud-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.dc = DeployerClient(token=token)

    def _init_nyan(self):
        """
        Provides self.nyan object with nyanbot client
        :return: None
        """
        from afisha.infra.libs.nyan import NyanClient
        self.nyan = NyanClient()

    def _init_u(self):
        """ Provides self.u object with U client """
        from afisha.infra.libs.u import UApiClient
        self.u = UApiClient()

    def _init_rc(self):
        """ Provides self.rc object with ReleaseApi client """
        from afisha.infra.libs.releases import ReleasesApiClient
        self.rc = ReleasesApiClient()

    def _init_ac(self, user):
        """ Provides self.ac object with ArcanumApi client """
        vault = "{}.arc-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.ac = AfishaArcanumApi(token=token)

    @staticmethod
    def find_task(tasks_ids, key, value):
        logging.debug("Searching %s parameter key with %s value in % task ids", key, value, tasks_ids)
        for task_id in tasks_ids:
            task_ = sdk2.Task[task_id]
            if getattr(task_.Parameters, key, None) == value:
                logging.debug("Found in %s", task_id)
                return task_
            logging.debug("Not found in %s", task_id)
        logging.debug("Not found in %s tasks", tasks_ids)
        return None

    @property
    def subtasks(self):
        if self.Context.subtasks is ctm.NotExists:
            return {}
        return self.Context.subtasks

    def task(self, task_type, **kwargs):
        kwargs["UseLastBinary"] = kwargs.get("UseLastBinary", self.Parameters.UseLastBinary)
        if kwargs["UseLastBinary"]:
            kwargs["ReleaseType"] = kwargs.get("ReleaseType", self.Parameters.ReleaseType)
        task = sdk2.Task[task_type](self, **kwargs)
        return task

    def subtasks_run(self, tasks, statuses=DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True, timeout=None):
        from afisha.infra.libs.base.funcs import to_sandbox_name
        if not self.subtasks:
            self.Context.subtasks = {}
        enqueued = []
        for task in tasks:
            task.enqueue()
            task_name = to_sandbox_name(type(task).__name__)
            if task_name not in self.subtasks:
                self.subtasks[task_name] = []
            if task.id not in self.subtasks[task_name]:
                self.subtasks[task_name].append(task.id)
            enqueued.append(task.id)
        if statuses:
            raise sdk2.WaitTask(enqueued, statuses, wait_all=wait_all, timeout=timeout)
