import logging
import re

import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task

from sandbox.projects.mediabilling.deploy.util import TaskHelper, MBCONFIG


class MediabillingBuildDocker(binary_task.LastBinaryTaskRelease, TaskHelper):
    """ Build Mediabilling Docker based on the revision number """
    fail_on_any_error = True

    class Requirements(sdk2.Task.Requirements):
        disk_space = 75 * 1024  # in megabytes

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 40 * 60
        description = "Build Yandex.Mediabilling Dockers"

        url = sdk2.parameters.String("The arcadia url",
                                     description=MBCONFIG.arcadia_description,
                                     required=True)

        patch = sdk2.parameters.String("The patch file to apply")
        patch_over_revision = sdk2.parameters.Integer("The patch is from this revision")

        with sdk2.parameters.CheckGroup("Apps", required=True) as targets:
            targets.choices = (
                ("yd-api", "yd-api"),
                ("yd-admin", "yd-admin"),
                ("yd-external-api", "yd-external-api"),
                ("yd-gw", "yd-gw"),
                ("yd-promotion-offers", "yd-promotion-offers"),
                ("yd-promotion-offers-admin", "yd-promotion-offers-admin"),
                ("yd-connectors", "yd-connectors"),
                ("yd-fast-prices", "yd-fast-prices"),
                ("yd-worker", "yd-worker"),
            )

        with sdk2.parameters.Output():
            version = sdk2.parameters.String("Docker version")
            branch = sdk2.parameters.String("The branch built from")
            revision = sdk2.parameters.Integer("The revision built from")

    def create_subtasks(self):
        task_class = sdk2.Task["YA_PACKAGE"]
        self.Context.subtasks = []
        arcadia_patch = "zipatch:" + str(self.Parameters.patch) if self.Parameters.patch else None
        for target in self.Parameters.targets:
            subtask = task_class(self,
                                 description=self.Parameters.description,
                                 checkout="True",
                                 packages=MBCONFIG.dockers[target]["config"],
                                 package_type="docker",
                                 docker_push_image="True",
                                 docker_registry="registry.yandex.net",
                                 docker_image_repository="mediabilling",
                                 docker_user="robot-mediabilling",
                                 docker_token_vault_name="robot-mb-docker",
                                 docker_build_network="host",
                                 ya_yt_store="True",
                                 use_aapi_fuse="True",
                                 use_arc_instead_of_aapi="True",
                                 arcadia_patch=arcadia_patch,
                                 build_system="ya",
                                 checkout_arcadia_from_url=str(self.Parameters.url),
                                 force_build_depends="True",
                                 checkout_mode="manual")
            self.Context.subtasks.append(subtask.enqueue().id)
        raise sdk2.WaitTask(self.Context.subtasks, Status.Group.FINISH + Status.Group.BREAK, wait_all=True, timeout=None)

    def has_docker(self):
        for target in self.Parameters.targets:
            if target in MBCONFIG.dockers:
                return True
        return False

    def check_subtasks(self):
        for subtask_id in self.Context.subtasks:
            if sdk2.Task[subtask_id].status != Status.SUCCESS:
                raise errors.TaskFailure()

        resource_version = sdk2.Resource["YA_PACKAGE"].find(task=sdk2.Task[self.Context.subtasks[0]]).first().resource_version

        logging.debug("resource_version: %s", resource_version)

        pattern = re.compile("registry.yandex.net/.*:(.+)")
        self.Parameters.version = pattern.findall(resource_version)[0]

    def on_execute(self):
        super(MediabillingBuildDocker, self).on_execute()
        branch, revision = TaskHelper.extract_branch_and_revision(self.Parameters.url)
        self.Parameters.branch = branch
        self.Parameters.revision = revision
        if not self.has_docker():
            return

        if MBCONFIG.is_dev:
            self.Context.subtasks = [746583091]
            self.Context.save()
        else:
            with self.memoize_stage.create_subtask:
                # will run only once
                self.create_subtasks()

        self.check_subtasks()

