# coding: utf-8
import json
import logging

import sandbox.sdk2 as sdk2
from sandbox.projects.Afisha.deploy.common.AfishaDeploy import AfishaDeploy
from sandbox.projects.common import binary_task
from sandbox.projects.music.deployment.helpers.DashboardUpdater import DashboardUpdater

from sandbox.projects.mediabilling.deploy.MediabillingBuildDocker import MediabillingBuildDocker
from sandbox.projects.mediabilling.deploy.MediabillingBuildJars import MediabillingBuildJars
from sandbox.projects.mediabilling.deploy.util import MBCONFIG, TaskHelper, Deployment, subtaskable
from sandbox.projects.mediabilling.deploy.util.NannyDockerUpdater import NannyDockerUpdater


class MediabillingDeploy(binary_task.LastBinaryTaskRelease, TaskHelper):
    """ Test task """

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 10 * 60
        description = "Deploy Mediabilling Api"
        revision = sdk2.parameters.Integer("Покатуемая ревизия", required=True)
        branch = sdk2.parameters.String("Покатуемая ветка", required=False)

        with sdk2.parameters.RadioGroup("Куда покатим") as mode:
            mode.values['deploy-testing'] = mode.Value('👗 в testing')
            mode.values['deploy-prod'] = mode.Value('👙 в прод!')

        with sdk2.parameters.CheckGroup('Что покатим', required=True) as targets:
            targets.choices = (
                ('yd-api', 'yd-api'),
                ('yd-promotion-offers', 'yd-promotion-offers'),
                ('yd-promotion-offers-admin', 'yd-promotion-offers-admin'),
                ('yd-admin', 'yd-admin'),
                ('yd-connectors', 'yd-connectors'),
                ('yd-gw', 'yd-gw'),
                ('yd-external-api', 'yd-external-api'),
                ('yd-worker', 'yd-worker'),
                ('yd-fast-prices', 'yd-fast-prices'),
                ('connector', 'music-connector'),
            )

        with sdk2.parameters.Output():
            dashboards = sdk2.parameters.String("Running dashboard ids")
            yd_envs = sdk2.parameters.String("Deploying yd environments with docker tags")

    @staticmethod
    def _filter_targets(target_type, targets):
        """
        Filter targets from parameters by its type
        :param target_type: oneOf('nanny_jar', 'nanny_docker', 'yd')
        :param targets: list of targets, list(str)
        :return: list of targets, list(str)
        """
        targets_filter_map = {
            'nanny_jar': lambda x: [y for y in x if y not in MBCONFIG.dockers],
            'nanny_docker': lambda x: [y for y in x if y in MBCONFIG.dockers and y not in MBCONFIG.yd_services],
            'yd': lambda x: [y for y in x if y in MBCONFIG.yd_services]
        }
        try:
            return targets_filter_map[target_type](targets)
        except KeyError:
            raise NotImplementedError('Dont know how process %s target_type, known types: %s' % (target_type, targets_filter_map.keys()))

    @subtaskable(True)
    def build_jars(self):
        if self.Parameters.branch is not None:
            url = 'arcadia:/arc/branches/mediabilling/common/{}/arcadia@{}'.format(self.Parameters.branch,
                                                                                   self.Parameters.revision)
        else:
            url = 'arcadia:/arc/trunk/arcadia@{}'.format(self.Parameters.revision)
        self.enqueue_subtask(MediabillingBuildJars,
                             binary_release=self.Parameters.binary_executor_release_type,
                             url=url,
                             jdk11=True,
                             yt_cache=True,
                             description='Building {}'.format(url))
        branch_and_revision = '{}/{}'.format(self.Parameters.branch, self.Parameters.revision)
        self.nyan_safe(u'🏎🏎🏎 Собираю {} для рецепта {}. С поул-позишн стартует {}'.format(
            branch_and_revision, self.Parameters.mode, self.author))

    @subtaskable(True)
    def build_docker(self):
        if self.Parameters.branch is not None:
            url = 'arcadia:/arc/branches/mediabilling/common/{}/arcadia@{}'.format(self.Parameters.branch,
                                                                                   self.Parameters.revision)
        else:
            url = 'arcadia:/arc/trunk/arcadia@{}'.format(self.Parameters.revision)
        self.enqueue_subtask(MediabillingBuildDocker,
                             binary_release=self.Parameters.binary_executor_release_type,
                             url=url,
                             targets=self._filter_targets('nanny_docker', self.Parameters.targets) + self._filter_targets('yd', self.Parameters.targets),
                             description='Building {}'.format(url))

    def _nanny_update(self, branch_and_revision, oauth_token, mediabilling_build_jars, mediabilling_build_docker):
        dockerUpdater = NannyDockerUpdater()
        for target in self._filter_targets('nanny_docker', self.Parameters.targets):
            dockerUpdater.updateDocker(
                oauth_token,
                target,
                self.Parameters.mode,
                MBCONFIG.dockers[target]['path'] + mediabilling_build_docker.Parameters.version,
                'Update to {} (task id {})'.format(branch_and_revision, mediabilling_build_docker.id)
            )

        updater = DashboardUpdater(oauth_token, MBCONFIG.build_jars_task_type)
        result = {}
        for target in self._filter_targets('nanny_docker', self.Parameters.targets):
            logging.debug('Processing target: %s', target)
            updater.reject_old_deployment(self.author, target, [self.Parameters.mode])
            result[target] = updater.run_dashboard(target, [self.Parameters.mode])
        for target in self._filter_targets('nanny_jar', self.Parameters.targets):
            logging.debug('Processing target: %s', target)
            result[target] = updater.set_and_update(self.author,
                                                    target,
                                                    [self.Parameters.mode],
                                                    mediabilling_build_jars.id,
                                                    branch_and_revision)
        logging.debug('_nanny_update result: %s', result)
        return result

    @subtaskable
    def _yd_update_wait(self, afisha_deploy):
        """ Just wait until afisha_deploy tasks finish """
        return

    def _yd_update(self, branch_and_revision, mediabilling_build_docker):
        comment = 'Update to {} (task id {})'.format(branch_and_revision, mediabilling_build_docker.id)
        result = {}
        for target in self._filter_targets('yd', self.Parameters.targets):
            logging.debug('Processing target: %s', target)
            try:
                yd_envs = MBCONFIG.yd_services[target][self.Parameters.mode]
            except KeyError:
                raise RuntimeError('Target %s not defined in MBCONFIG.yd_services' % target)
            for yd_env in yd_envs:
                logging.debug('Enqueue subtask for env: %s', yd_env)
                self.enqueue_subtask(AfishaDeploy,
                                     owner=self.Parameters.owner,
                                     priority=self.Parameters.priority,
                                     description=comment,
                                     kill_timeout=120,
                                     qloud_environments=yd_env,
                                     registry_url=':{}'.format(mediabilling_build_docker.Parameters.version),
                                     tgt_state='DEPLOYED',
                                     comment=comment,
                                     deploy_with_robot=True,
                                     deployer_username=MBCONFIG.deployer_username)
                result[yd_env] = {'tag': mediabilling_build_docker.Parameters.version, 'ready': False}
        self._yd_update_wait()
        logging.debug('_yd_update result: %s', result)
        return result

    def _deploy(self, oauth_token, branch_and_revision, mediabilling_build_jars, mediabilling_build_docker):
        nanny_result = self._nanny_update(branch_and_revision, oauth_token, mediabilling_build_jars, mediabilling_build_docker)
        yd_result = self._yd_update(branch_and_revision, mediabilling_build_docker)
        self.Parameters.dashboards = json.dumps(nanny_result)
        self.Parameters.yd_envs = json.dumps(yd_result)
        return nanny_result, yd_result

    @subtaskable(True)
    def deploy(self, oauth_token, branch_and_revision, mediabilling_build_jars, mediabilling_build_docker):
        self._deploy(oauth_token, branch_and_revision, mediabilling_build_jars, mediabilling_build_docker)
        self.nyan_safe(u'🛸🛸🛸 Запущен {} рецепт {} ({}). Красную кнопку нажал {}'.format(
            self.Parameters.mode, ', '.join(self.Parameters.targets), branch_and_revision, self.author))

    def deploy_cached(self, oauth_token, branch_and_revision, mediabilling_build_jars, mediabilling_build_docker):
        self._deploy(oauth_token, branch_and_revision, mediabilling_build_jars, mediabilling_build_docker)
        self.nyan_safe(u'🚀🚀🚀 Запущен {} рецепт {} ({}). Пилот ракеты {}'.format(
            self.Parameters.mode, ', '.join(self.Parameters.targets), branch_and_revision, self.author))

    def on_execute(self):
        super(MediabillingDeploy, self).on_execute()
        secret = sdk2.yav.Secret(MBCONFIG.yav_token)
        oauth_token = secret.data()[MBCONFIG.token_name]
        branch_and_revision = '{}/{}'.format(self.Parameters.branch, self.Parameters.revision)
        self.check_authorization(self.author, oauth_token, MBCONFIG.auth_build)

        mediabilling_build_jars = Deployment.find_build_jars_task(
            self.Parameters.revision,
            self.Parameters.branch if self.Parameters.branch is not None else self.Parameters.branch)
        mediabilling_build_docker = Deployment.find_build_docker_task(
            MediabillingBuildDocker,
            self.Parameters.revision,
            self.Parameters.branch if self.Parameters.branch is not None else self.Parameters.branch,
            [t for t in self.Parameters.targets if t in MBCONFIG.dockers])

        # debug
        if MBCONFIG.is_dev:
            mediabilling_build_jars = sdk2.Task.find(id=742777695).first()
            if not mediabilling_build_docker:
                self.build_docker()
                self.deploy(
                    oauth_token,
                    branch_and_revision,
                    mediabilling_build_jars)
            else:
                mediabilling_build_jars = sdk2.Task.find(id=742777695).first()
                self.deployCached(
                    oauth_token,
                    branch_and_revision,
                    mediabilling_build_jars,
                    mediabilling_build_docker)
            return

        if not mediabilling_build_jars or not mediabilling_build_docker:
            self.build_jars()
            self.build_docker()
            self.deploy(oauth_token, branch_and_revision)
        else:
            self.deploy_cached(oauth_token, branch_and_revision, mediabilling_build_jars, mediabilling_build_docker)
