# coding: utf-8

import json
import os
from collections import Counter

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.projects.common import task_env, binary_task
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper

from sandbox.projects.mediabilling.deploy.util import TaskHelper, MBCONFIG


class MediabillingReleaseTests(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        description = "Run E2E tests"
        arcadia_url = sdk2.parameters.String("The arcadia url",
                                             default="arcadia:/arc/trunk/arcadia",
                                             description=MBCONFIG.arcadia_description,
                                             required=True)

        arcadia_patch = sdk2.parameters.String("The patch file to apply", required=False)
        issue = sdk2.parameters.String("Startrek issue to inform to", required=False)

    def create_subtasks(self, token):
        arcadia_patch = "zipatch:" + str(self.Parameters.arcadia_patch) if self.Parameters.arcadia_patch else None

        e2e_suites = self.collect_test_suites(token)
        self.Context.suites = e2e_suites
        self.Context.subtasks = [self.create_subtask(arcadia_patch, suite).enqueue().id for suite in e2e_suites]
        self.Context.save()
        raise sdk2.WaitTask(self.Context.subtasks, Status.Group.FINISH + Status.Group.BREAK, wait_all=True)

    def collect_test_suites(self, token):
        e2e_suites = []
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url, arc_oauth_token=token) as arcadia_path:
            for subdir, _, _ in os.walk(os.path.join(arcadia_path, "media-billing", "mediabilling-test-end-to-end")):
                ya_make_path = os.path.join(subdir, "ya.make")
                if os.path.isfile(ya_make_path) and self.is_test_ya_make(ya_make_path):
                    e2e_suites.append(os.path.relpath(subdir, arcadia_path))
        return e2e_suites

    @staticmethod
    def is_test_ya_make(filename):
        with open(filename, "r") as f:
            return "JUNIT5()" in f.read()

    def create_subtask(self, arcadia_patch, suite):
        task_class = sdk2.Task["YA_MAKE_2"]
        return task_class(self,
                          description="E2E mediabilling test: " + suite,
                          checkout="True",
                          targets=suite,
                          test="True",
                          arcadia_patch=arcadia_patch,
                          test_log_level="debug",
                          test_failure_code="10",
                          test_retries="1",
                          build_system="ya",
                          allure_report="True",
                          allure_report_ttl=21,
                          checkout_arcadia_from_url=str(self.Parameters.arcadia_url),
                          force_build_depends="True",
                          checkout_mode="auto",
                          junit_report="True",
                          use_aapi_fuse="True",
                          aapi_fallback="True",
                          keep_on="True",
                          ram_drive_size=1024,
                          disable_test_timeout="True",
                          sandbox_tags="IPV6 & LINUX_XENIAL")

    def check_subtask(self, token):
        st = StartrekHelper(token)

        if self.Parameters.issue:
            result_table = '#|\n'
            for suite, subtask_id in zip(self.Context.suites, self.Context.subtasks):
                subtask = sdk2.Task.find(id=subtask_id).first()
                report_resource = sdk2.Resource["ALLURE_REPORT"].find(task=subtask).first()
                allure_report_root = report_resource.http_proxy
                statuses_by_category = self.load_test_statuses(str(sdk2.ResourceData(report_resource).path))

                total_counter = reduce(lambda acc, c: acc + c, statuses_by_category.itervalues(), Counter())
                regress_counter = statuses_by_category.get("Regress", Counter())

                result_table += (u'||!!({suite_color}){suite}!!'
                                 u'|(({allure_report_root}/index.html report))'
                                 u'|((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))'
                                 u'|{total_statuses}'
                                 u'|Regress: {regress_statuses}'
                                 u'||\n').format(suite_color='red' if total_counter["failed"] > 0 else 'green',
                                                 suite=suite,
                                                 allure_report_root=allure_report_root,
                                                 task_id=subtask_id,
                                                 total_statuses=self.build_statuses_string(total_counter),
                                                 regress_statuses=self.build_statuses_string(regress_counter))

            result_table += '|#'
            st.add_comment(self.Parameters.issue, result_table)

    @staticmethod
    def build_statuses_string(counter):
        return '!!(green){passed}!!/!!(grey){skipped}!!/!!(red){failed}!!'.format(
            passed=counter["passed"],
            skipped=counter["skipped"],
            failed=counter["failed"])

    @staticmethod
    def load_test_statuses(report_root):
        path = os.path.join(report_root, "data", "behaviors.json")
        with open(path) as f:
            return MediabillingReleaseTests.count_test_statuses(json.load(f)['children'])

    @staticmethod
    def count_test_statuses(tests, storage=None, category_name="Top"):
        if storage is None:
            storage = {}
        counter = Counter()

        for test in tests:
            if "status" in test:
                counter.update([test["status"]])
            else:
                MediabillingReleaseTests.count_test_statuses(test["children"], storage, test["name"])

        storage[category_name] = counter
        return storage

    def on_execute(self):
        super(MediabillingReleaseTests, self).on_execute()
        secret = sdk2.yav.Secret(MBCONFIG.yav_token)
        oauth_token = secret.data()[MBCONFIG.token_name]
        with self.memoize_stage.create_subtask:
            # will run only once
            self.create_subtasks(oauth_token)

        self.check_subtask(oauth_token)
