# coding: utf-8
import logging
import re
from time import time

import requests
import sandbox.common.types.task as ctt
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.common import binary_task
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.release_machine.tasks.CreateBranchOrTag import CreateBranchOrTag

from sandbox.projects.mediabilling.deploy.MediabillingDeploy import MediabillingDeploy
from sandbox.projects.mediabilling.deploy.MediabillingWatchDeployment import MediabillingWatchDeployment
from sandbox.projects.mediabilling.deploy.util import (MBCONFIG, subtaskable, TaskHelper, ArcadiaHelper)

TRUNK = 'trunk'
URL_RE = re.compile(r'^arcadia:/arc/(?:trunk|branches/(mediabilling/common|music)/(stable-\d+))/arcadia(?:@(\d+))?$')


class MediabillingStartDeployment(binary_task.LastBinaryTaskRelease, TaskHelper):
    """ Start Mediabilling deployment, instead of the testenv """
    fail_on_any_error = True

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]
        cores = 1

        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name="mediabilling_deployment")
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 10 * 60  # real time ~1 min
        description = "Build and deploy Mediabilling"

        with sdk2.parameters.RadioGroup("Deployment mode") as mode:
            mode.values['standard'] = mode.Value('Standard: make a branch, build, deploy to testing and load')
            mode.values['hotfix'] = mode.Value('Hotfix: build and deploy to stable')
            mode.values['stable'] = mode.Value('Release: deploy to prod')

        with mode.value['standard']:
            # also: self.Context.url <- url with a revision set
            force_create_branch = sdk2.parameters.Bool('Создать тикет по ветке')
            standard_url = sdk2.parameters.String("URL to build from",
                                                  default=MBCONFIG.arcadia_trunk,
                                                  description=MBCONFIG.arcadia_description,
                                                  required=True)

        with mode.value['hotfix']:
            hotfix_ticket = sdk2.parameters.String("Тикет с описанием необходимости хотфикса",
                                                   default="MEDIABILLING-4321",
                                                   description="Тикет должен быть заапрувлен комментарием "
                                                               "или устно еще одним человеком",
                                                   required=True)
            hotfix_url = sdk2.parameters.String("URL to build from",
                                                default=MBCONFIG.arcadia_branch.format('XXX'),
                                                description=MBCONFIG.arcadia_description,
                                                required=True)
            with sdk2.parameters.CheckGroup('Что покатим', required=True) as targets:
                targets.choices = (
                    ('yd-api', 'yd-api'),
                    ('yd-promotion-offers', 'yd-promotion-offers'),
                    ('yd-promotion-offers-admin', 'yd-promotion-offers-admin'),
                    ('yd-worker', 'yd-worker'),
                    ('yd-admin', 'yd-admin'),
                    ('yd-connectors', 'yd-connectors'),
                    ('yd-gw', 'yd-gw'),
                    ('yd-external-api', 'yd-external-api'),
                    ('yd-fast-prices', 'yd-fast-prices'),
                    ('connector', 'music-connector'),
                )

        with mode.value['stable']:
            stable_task = sdk2.parameters.Task("{} task".format(MBCONFIG.build_jars_task_type),
                                               task_type=MBCONFIG.build_jars_task_type)

        with sdk2.parameters.Group("Specific settings", collapse=True) as specific_settings_block:
            allow_drills_deploy = sdk2.parameters.Bool("Allow deploy during drills ", default=False)

        with sdk2.parameters.Output():
            branch = sdk2.parameters.String("The branch name")
            revision = sdk2.parameters.Integer("Build revision")
            issue = sdk2.parameters.String("Startrek issue")

    def check_drills(self):
        # https://infra.yandex-team.ru/services/154/events
        infra_drills_config = {
            "enviroment_id": 204,
            "service_id": 154
        }
        infra_url = "https://infra-api.yandex-team.ru/v1/events"

        current_timestamp = int(time())
        current_timestamp_plus_two_hours = current_timestamp + 7200

        params = {"serviceId": infra_drills_config["service_id"],
                  "environmentId": infra_drills_config["enviroment_id"],
                  "from": current_timestamp,
                  "to": current_timestamp_plus_two_hours}
        if not self.Parameters.allow_drills_deploy:
            try:
                r = requests.get(infra_url, params=params, verify=False)
            except Exception as e:
                logging.error('Check drills failed: {}'.format(e))
                return True
            if r.json():
                logging.error('Found drills, stopping release')
                return True
            else:
                return False
        return False

    @subtaskable(True)
    def process_input_parameters_standard(self, token, _=None):
        match = URL_RE.match(str(self.Parameters.standard_url))
        if not match:
            raise errors.TaskFailure('The URL is in incompatible format')
        project_tmp, branch, revision = match.groups()

        if revision:
            self.Context.revision = int(revision)
            self.Context.url = str(self.Parameters.standard_url)
        else:
            self.Context.revision = ArcadiaHelper.get_latest_affected_revision(
                self.Parameters.standard_url + '/media-billing')
            self.Context.url = '{}@{}'.format(self.Parameters.standard_url, self.Context.revision)

        self.Context.branch = branch or TRUNK

        self.Context.mode = 'deploy-testing'
        self.Context.targets = ['music-connector',
                                'yd-api', 'yd-admin', 'yd-promotion-offers', 'yd-external-api',
                                'yd-promotion-offers-admin', 'yd-gw', 'yd-connectors', 'yd-fast-prices', 'yd-worker']

        st = StartrekHelper(token, MBCONFIG.queue)
        if self.Context.branch == TRUNK:
            if len(st.find_nonclosed_issues()) > 0:
                raise errors.TaskFailure('Cannot create a new release while there is a pending release')

    @subtaskable(True)
    def process_input_parameters_hotfix(self, token, _=None):
        match = URL_RE.match(str(self.Parameters.hotfix_url))
        if not match:
            raise errors.TaskFailure('The url is in incompatible format')
        project_tmp, branch, revision = match.groups()

        if not branch:
            raise errors.TaskFailure('Hotfixes can only be deployed from branches')

        if revision:
            self.Context.revision = int(revision)
        else:
            raise errors.TaskFailure('A hotfix must have revision')

        self.Context.branch = branch
        self.Context.revision = int(revision)
        self.Context.url = self.Parameters.hotfix_url
        self.Context.mode = 'deploy-prod'
        self.Context.targets = self.Parameters.targets

        startrek = StartrekHelper(token, MBCONFIG.queue)
        issue = startrek.find_issue_for_branch(self.Context.branch)
        self.Context.issue = issue.key

    @subtaskable(True)
    def process_input_parameters_stable(self, _=None):
        if str(self.Parameters.stable_task.type) != MBCONFIG.build_jars_task_type:
            raise errors.TaskFailure('The task type must be a {}, not {}'.format(MBCONFIG.build_jars_task_type,
                                                                                 self.Parameters.stable_task.type))

        self.Context.branch = self.Parameters.stable_task.Parameters.branch
        self.Context.revision = self.Parameters.stable_task.Parameters.revision
        self.Context.url = 'arcadia:/arc/branches/mediabilling/common/{}/arcadia@{}' \
            .format(self.Context.branch, self.Context.revision)
        self.Context.mode = 'deploy-prod'
        self.Context.targets = ['music-connector',
                                'yd-api', 'yd-admin', 'yd-promotion-offers', 'yd-external-api',
                                'yd-promotion-offers-admin', 'yd-gw', 'yd-connectors', 'yd-fast-prices', 'yd-worker']

    @subtaskable(True)
    def create_branch(self, _=None):
        if self.Context.branch == TRUNK:
            self.nyan_safe(u'🔪🔪🔪 Отрезаю релизную ветку. Ты красавчик, если успел закоммитить до ревизии {}'
                           .format(self.Context.revision))
            self.enqueue_subtask(CreateBranchOrTag,
                                 component_name='mediabilling',
                                 tag_or_branch_mode='trunk->branch',
                                 revision_for_trunk=self.Context.revision,
                                 description='Branching from revision {}'.format(self.Context.revision))
        else:
            self.nyan_safe(u'🧞‍♂️🧞‍♂️🧞‍♂️ Получил указание докатить релизную ветку {}/{} от {}'
                           .format(self.Context.branch, self.Context.revision, self.author))

    @subtaskable(True)
    def create_startrek_issue(self, token, create_branch_or_tag, _=None):
        startrek = StartrekHelper(token, MBCONFIG.queue)
        if create_branch_or_tag or self.Parameters.force_create_branch:
            if not self.Parameters.force_create_branch:
                self.Context.branch = create_branch_or_tag.Parameters.result_path.split('/')[-1]
                self.Context.revision = create_branch_or_tag.Parameters.result_revision
            # update build url to the new version
            branch_no = self.Context.branch.split('-')[1]

            self.Context.url = MBCONFIG.arcadia_branch.format(branch_no) + '@' + str(self.Context.revision)

            arcadia = ArcadiaHelper(startrek)
            parsed_changelog, commits_without_task, first_rev, _ = arcadia.prepare_and_extract_latest_changelog(
                self.Context.branch)
            on_duty = self.on_duty_backend(token)
            on_duty_testing = self.on_duty_testing(token)
            text = arcadia.render_changelog(parsed_changelog=parsed_changelog,
                                            commits_without_tasks=commits_without_task,
                                            from_revision=first_rev,
                                            to_revision=self.Context.revision,
                                            branch=self.Context.branch,
                                            on_duty=on_duty,
                                            on_duty_testing=on_duty_testing)
            issue = startrek.create_issue(
                MBCONFIG.issue_summary(self.Context.revision, self.Context.branch),
                text,
                MBCONFIG.checklist,
                assignee=on_duty)
            self.nyan_safe(u'🖇🖇🖇 Создан релизный тикет {}'.format(MBCONFIG.markdown_startrek(issue.key)))
        else:
            issue = startrek.find_issue_for_branch(self.Context.branch)
        self.Context.issue = issue.key
        return issue

    @subtaskable(True)
    def initiate_deployment(self, _=None):
        if self.Context.mode == 'deploy-prod':
            if self.check_drills():
                message = u"""
                Релиз не может поехать в stable, так как запланированы учения в ДЦ в следующие 2 часа
                """
                raise errors.TaskError(message)
        self.enqueue_subtask(MediabillingDeploy,
                             binary_release=self.Parameters.binary_executor_release_type,
                             branch=self.Context.branch,
                             revision=self.Context.revision,
                             mode=self.Context.mode,
                             targets=self.Context.targets,
                             description='Initiating {} deployment'.format(self.Context.url))

    def extract_hotfix_changelog(self, startrek, token):
        arcadia = ArcadiaHelper(startrek)
        issue = startrek.get_issue(self.Context.issue)
        revision, branch = startrek.summary_revision_and_branch(issue.summary)
        path = arcadia.get_url(self.Context.branch, subdir='arcadia/media-billing')
        tasks, commits, last_rev = arcadia.extract_changelog_path(path, revision)
        previous_branch = arcadia.find_previous_branch(self.Context.branch)
        logging.info('Current branch: {}, previous: {}'.format(self.Context.branch, previous_branch))
        first_rev = arcadia.branch_to_revision(previous_branch)
        logging.info('Render hotfix changelog r%s:r%s', first_rev, last_rev)
        text = arcadia.render_changelog(parsed_changelog=tasks,
                                        commits_without_tasks=commits,
                                        from_revision=first_rev,
                                        to_revision=last_rev,
                                        branch=branch,
                                        on_duty=self.on_duty_backend(token),
                                        on_duty_testing=self.on_duty_testing(token),
                                        hotfix=True)
        return text

    def mark_finalization_in_startrek(self, token, target, build_task, deploy_task, changelog=False, reason=None):
        st = StartrekHelper(token)
        text = (u'Запущена выкатка {target}\n\n'
                u'Ветка: {branch}\n'
                u'Ревизия: {revision}\n'
                u'Сборочная задача: {build_task}\n'
                u'Выкладочная задача: {deploy_task}\n').format(
            target=target,
            branch=self.Context.branch,
            revision=self.Context.revision,
            build_task=MBCONFIG.wikiformat_sandbox(str(build_task)),
            deploy_task=MBCONFIG.wikiformat_sandbox(str(deploy_task)),
        )

        if changelog:
            text += "\n" + self.extract_hotfix_changelog(st, token)

        if reason:
            text += u'\n\nПричина выкатки: {}'.format(reason)

        st.add_comment(self.Context.issue, text)
        st.set_issue_release_notes(self.Context.issue, str(build_task))

    def enqueue_watcher(self, mediabilling_deploy, target, reason='_'):
        self.enqueue_subtask(MediabillingWatchDeployment,
                             binary_release=self.Parameters.binary_executor_release_type,
                             dashboards=mediabilling_deploy.Parameters.dashboards,
                             yd_envs=mediabilling_deploy.Parameters.yd_envs,
                             branch=self.Context.branch,
                             revision=self.Context.revision,
                             issue=self.Context.issue,
                             target=target,
                             reason=reason)

    @subtaskable(True)
    def finalize_standard(self, token, mediabilling_deploy, _=None):
        subtasks = sdk2.Task.find(parent=mediabilling_deploy).limit(25)
        subtask_dict = dict((str(task.type), task) for task in subtasks if task.status != ctt.Status.DRAFT)
        mediabilling_build_jars = subtask_dict.get('MEDIABILLING_BUILD_JARS', None)
        task_id = mediabilling_build_jars.id if mediabilling_build_jars else self.Context.done_task
        self.mark_finalization_in_startrek(token,
                                           u'в тестинг',
                                           task_id,
                                           mediabilling_deploy.id)
        self.enqueue_watcher(mediabilling_deploy, 'testing', 'Релиз!Релиз!Релиз!')

    @subtaskable(True)
    def finalize_hotfix(self, token, mediabilling_deploy, _=None):
        hotfix_ticket = self.Parameters.hotfix_ticket
        subtasks = sdk2.Task.find(parent=mediabilling_deploy).limit(25)
        subtask_dict = dict((str(task.type), task) for task in subtasks if task.status != ctt.Status.DRAFT)
        mediabilling_build_jars = subtask_dict.get('MEDIABILLING_BUILD_JARS', None)
        task_id = mediabilling_build_jars.id if mediabilling_build_jars else self.Context.done_task
        self.mark_finalization_in_startrek(token,
                                           u'ХОТФИКСА',
                                           task_id,
                                           mediabilling_deploy.id,
                                           changelog=True,
                                           reason='Хотфикс без причины признак медиабиллинга кончины. '
                                                  'Хотфиксный тикет: ' + hotfix_ticket)
        self.enqueue_watcher(mediabilling_deploy, 'hotfix', 'Хотфикс без причины признак медиабиллинга кончины. '
                                                            'Хотфиксный тикет: '
                             + MBCONFIG.markdown_startrek(hotfix_ticket))

    @subtaskable(True)
    def finalize_stable(self, token, mediabilling_deploy, _=None):
        task_id = self.Parameters.stable_task.id

        startrek = StartrekHelper(token, MBCONFIG.queue)
        issue = startrek.find_issue_for_branch(self.Context.branch)
        self.Context.issue = issue.key

        self.mark_finalization_in_startrek(token,
                                           u'в бой',
                                           task_id,
                                           mediabilling_deploy.id)
        self.enqueue_watcher(mediabilling_deploy, 'stable', 'Чак-чак и в продакшн')

    def on_execute(self):
        super(MediabillingStartDeployment, self).on_execute()
        self.check_there_is_only_one_such_task()

        secret = sdk2.yav.Secret(MBCONFIG.yav_token)
        token = secret.data()[MBCONFIG.token_name]

        if self.Parameters.mode == 'standard':
            self.process_input_parameters_standard(token)
            self.create_branch()
            self.create_startrek_issue(token)
            self.initiate_deployment()
            self.finalize_standard(token)
        elif self.Parameters.mode == 'hotfix':
            self.process_input_parameters_hotfix(token=token)
            self.initiate_deployment()
            self.finalize_hotfix(token)
        elif self.Parameters.mode == 'stable':
            self.process_input_parameters_stable()
            self.initiate_deployment()
            self.finalize_stable(token)
        else:
            raise errors.TaskError('Unknown mode {}'.format(self.Parameters.mode))

        self.Parameters.branch = self.Context.branch
        self.Parameters.revision = self.Context.revision
        self.Parameters.issue = self.Context.issue
