# coding: utf-8
import json
import os

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper

from sandbox.projects.mediabilling.deploy.util import TaskHelper, MBCONFIG


class MediabillingSwaggerCoverageReport(binary_task.LastBinaryTaskRelease, TaskHelper):
    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        description = "Run tests and sends swagger coverage report"

        arcadia_url = sdk2.parameters.String("The arcadia url",
                                             default="arcadia:/arc/trunk/arcadia",
                                             description=MBCONFIG.arcadia_description,
                                             required=True)

        arcadia_patch = sdk2.parameters.String("The patch file to apply", required=False)
        issue = sdk2.parameters.String("Startrek issue to inform to", required=False)

    def create_subtask(self):
        task_class = sdk2.Task["MEDIABILLING_YA_MAKE"]

        arcadia_patch = "zipatch:" + str(self.Parameters.arcadia_patch) if self.Parameters.arcadia_patch else None

        sub_task = task_class(self,
                              description="Mediabilling API tests",
                              checkout="True",
                              targets="media-billing",
                              test="True",
                              arcadia_patch=arcadia_patch,
                              test_tag="swagger_coverage",
                              test_log_level="debug",
                              test_failure_code="10",
                              test_retries="1",
                              build_system="ya",
                              checkout_arcadia_from_url=str(self.Parameters.arcadia_url),
                              force_build_depends="True",
                              checkout_mode="auto",
                              use_aapi_fuse="True",
                              aapi_fallback="True",
                              keep_on="True",
                              sandbox_tags="IPV6 & LINUX_XENIAL")
        sub_task.enqueue()
        self.Context.subtask = sub_task.id
        self.Context.save()
        raise sdk2.WaitTask(sub_task.id, Status.Group.FINISH + Status.Group.BREAK)

    def on_execute(self):
        super(MediabillingSwaggerCoverageReport, self).on_execute()
        with self.memoize_stage.create_subtask:
            # will run only once
            self.create_subtask()
        self.process_swagger_coverage_report()

    def process_swagger_coverage_report(self):
        secret = sdk2.yav.Secret(MBCONFIG.yav_token)
        oauth_token = secret.data()[MBCONFIG.token_name]
        st = StartrekHelper(oauth_token)

        sub_task = sdk2.Task.find(id=self.Context.subtask).first()
        build_output_resource = sdk2.Resource["BUILD_OUTPUT"].find(task=sub_task).first()
        build_output_proxy = build_output_resource.http_proxy
        path = sdk2.ResourceData(build_output_resource).path

        reports = []
        for root, _, _ in os.walk(str(path)):
            if root.endswith("swagger_coverage_report"):
                report_html_path = os.path.join(root, "swagger-coverage-report.html")
                results_json_path = os.path.join(root, "swagger-coverage-results.json")
                if os.path.exists(report_html_path) and os.path.exists(results_json_path):
                    reports.append({
                        'report_proxy_path': build_output_proxy + self.extract_path_in_http_proxy(report_html_path),
                        'suite_name': self.extract_test_suite_name(root),
                        'conditions_stats': self.extract_coverage_stats(results_json_path)
                    })

        if reports and self.Parameters.issue:
            message = self.build_st_message(reports)

            st.add_comment(self.Parameters.issue,
                           u'**Swagger coverage report**\n'
                           u'{message}'
                           u'\n'
                           u'Sandbox task: ((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))\n'
                           .format(message=message,
                                   task_id=self.id))

    @staticmethod
    def build_st_message(reports):
        msg = u'#|\n'
        msg += u'|| Test suite | Conditions coverage||\n'
        for report_data in reports:
            msg += u'|| (({report_proxy_path} {suite_name}))| {conditions_stats}||\n'.format(**report_data)
        msg += u'|#'
        return msg

    @staticmethod
    def extract_test_suite_name(report_dir):
        return os.path.basename(os.path.dirname(os.path.dirname(report_dir)))

    @staticmethod
    def extract_path_in_http_proxy(full_path):
        return full_path.split("/build", 1)[1]

    @staticmethod
    def extract_coverage_stats(json_path):
        with open(json_path) as f:
            coverage_json = json.load(f)
            condition_counter = coverage_json['conditionCounter']
            total = int(condition_counter['all'])
            covered = int(condition_counter['covered'])
            return '{}/{} ({:.2f}%)'.format(covered, total, 100 * (float(covered) / float(total)))
