# coding: utf-8
import datetime
import json
import logging

import sandbox.common.types.task as ctt
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.common.types.misc import NotExists
from sandbox.projects.Afisha.deploy.common.AfishaCheckDeploy import AfishaCheckDeploy
from sandbox.projects.common import binary_task
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper

from sandbox.projects.mediabilling.deploy.util import (MBCONFIG, TaskHelper, ArcadiaHelper)


class MediabillingWatchDeployment(binary_task.LastBinaryTaskRelease, TaskHelper):
    """Watch deployment process and inform on success"""

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 10 * 60
        description = "Подсматриваю за релизами медиабиллинга"

        dashboards = sdk2.parameters.String("Dashboard ids dict", required=True)
        yd_envs = sdk2.parameters.String("YD environments dict", required=False, default="{}")
        branch = sdk2.parameters.String("The deployment branch", required=True)
        revision = sdk2.parameters.Integer("The deployment revision", required=True)
        issue = sdk2.parameters.String("The deployment issue", required=True)
        reason = sdk2.parameters.String("The deployment reason", default="", required=True)

        with sdk2.parameters.RadioGroup("Deployment target", required=True) as target:
            target.values.testing = "testing"
            target.values.stable = "stable"
            target.values.hotfix = "hotfix"

    @staticmethod
    def dashboards_to_deployments(dashboards, yd_envs, environment):
        from afisha.infra.libs.deploy import DeployEnvInfo
        logging.debug('Generating message for environment %s with dashboards: %s, yd_envs: %s',
                      environment, dashboards, yd_envs)
        result = ''
        for service in MBCONFIG.dashboards:
            logging.debug('Processing %s service', service)
            if service in dashboards and environment in dashboards[service]:
                result += (
                    MBCONFIG.list_prefix +
                    u'[{}]({}){}\n'.format(
                        service,
                        MBCONFIG.deployment_url(service, dashboards[service][environment]['deployment']),
                        u' — готово' if dashboards[service][environment].get('is_ready') else ''
                    )
                )
        for environment, data in yd_envs.items():
            logging.debug('Processing %s service: %s', environment)
            result += (
                MBCONFIG.list_prefix +
                u'[{}]({}){}\n'.format(
                    environment,
                    DeployEnvInfo.from_envid(environment).url,
                    u' — готово' if data.get('ready') else ''
                )
            )
        return result

    def template_testing(self, deployments_testing):
        template = self.preprocess_template(u"""
            🥌🥌🥌 Катка {branch}/{revision} в тестинг:

            {reason}

            {deployments_testing}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=MBCONFIG.markdown_startrek(self.Parameters.issue),
                               deployments_testing=deployments_testing,
                               reason=self.Parameters.reason)

    def template_stable(self, stable):
        template = self.preprocess_template(u"""
            👁👁👁 Катка {branch}/{revision} в бой:

            {reason}

            {stable}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=MBCONFIG.markdown_startrek(self.Parameters.issue),
                               stable=stable,
                               reason=self.Parameters.reason)

    def template_hotfix(self, stable):
        template = self.preprocess_template(u"""
            👁👁👁 Катка {branch}/{revision} в бой:

            {reason}

            {stable}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=MBCONFIG.markdown_startrek(self.Parameters.issue),
                               stable=stable,
                               reason=self.Parameters.reason)

    def template_done(self, running_seconds):
        template = self.preprocess_template(u"""
            🌱🌱🌱 Катка {branch}/{revision} ({target}) завершена за {time} минут!
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               target=self.Parameters.target,
                               time=int(running_seconds / 60))

    def template_timed_out(self, running_seconds, on_duty):
        template = self.preprocess_template(u"""
            ❌❌❌ Катка {branch}/{revision} ({target}) не была завершена за {time} минут!
            @{on_duty}, посмотри плиз
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               target=self.Parameters.target,
                               time=int(running_seconds / 60),
                               on_duty=on_duty)

    def report_task_done(self, token):
        """
        Report task done to Startrek release ticket

        :param token:  Startrek client token
        """
        st = StartrekHelper(token)

        template = self.preprocess_template(u"""
            Катка в {env} завершена!\n\n
            Ветка: {branch}
            Ревизия: {revision}
        """.format(env=self.Parameters.target,
                   branch=self.Parameters.branch,
                   revision=self.Parameters.revision))

        st.add_comment(self.Parameters.issue, template)

    def deploy_tests(self):
        for test in MBCONFIG.deployment_tests:
            self.enqueue_subtask(test, binary_release=self.Parameters.binary_executor_release_type, issue=self.Parameters.issue)

    def report_task_failed(self, token, running_seconds):
        """
        Report task failed to Startrek release ticket

        :param running_seconds: Seconds elapsed since deployment started
        :param token:  Startrek client token
        """
        st = StartrekHelper(token)

        template = self.preprocess_template(u"""
            Катка в {env} не была завершена за {time} минут!\n\n
            Ветка: {branch}
            Ревизия: {revision}
        """.format(env=self.Parameters.target,
                   branch=self.Parameters.branch,
                   revision=self.Parameters.revision,
                   time=(running_seconds / 60)))

        st.add_comment(self.Parameters.issue, template)

    def set_issue_deployed_to_prod(self, token):
        st = StartrekHelper(token)
        st.set_issue_status(self.Parameters.issue, 'deployed')

    @staticmethod
    def summon_author_to_released_issue(startrek, issue_key, author, release_issue):
        if any(issue_key.startswith(queue + "-") for queue in ["PLUSCOMMON", "MEDIABILLING"]):
            comment = (u'Задача выкатилась в релизе {release_issue}\n'
                       u'Не забудь включить фичафлаги и проверить задачу в проде. '
                       u'Если всё ок - переведи задачу в статус "Закрыт"'
                       ).format(release_issue=release_issue)
            startrek.add_comment(issue_key, comment, author)

    def _check_nanny(self, dashboards, token):
        logging.debug('Checking nanny dashboards: %s', dashboards)
        nanny = NannyClient(MBCONFIG.nanny_api_url, token)
        all_ready = True

        for service, env_data in dashboards.items():
            logging.debug('Checking service %s with data %s', service, env_data)
            for env, data in env_data.items():
                taskgroup = data['taskgroup']
                is_ready = (nanny.get_taskgroup_status(taskgroup)['status'] == 'DONE')
                data['is_ready'] = is_ready
                logging.debug('Ready flag for %s in %s env: %s', service, env, is_ready)
                all_ready = all_ready & is_ready
                logging.debug('All ready flag now is: %s', all_ready)
        logging.debug('Processed dashboards result: %s', dashboards)
        return all_ready

    def _check_yd(self, environment_map):
        """
        Check if environment in YD is on target registry tag
        :param environment_map: {environment_id: {'tag': registry_tag, 'ready': boolean}}, dict; environment_id e.g. media-billing.internal-api.testing
        :return: True if all environments are on their registry_tag
        """
        from afisha.infra.libs.deploy import DeployEnvInfo
        logging.debug('Checking yd environments: %s', environment_map)
        running_tasks = []
        for environment, data in environment_map.items():
            logging.debug('Checking environment %s with data %s', environment, data)
            d_env = DeployEnvInfo.from_envid(environment)
            subtasks = self.subtasks.get("AFISHA_CHECK_DEPLOY", [])
            task = self.filter_tasks_by_param(subtasks, "environment", environment)
            if task:
                if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                    data['ready'] = True
                    continue
            else:
                task = self.enqueue_subtask(AfishaCheckDeploy,
                                            owner=self.Parameters.owner,
                                            priority=self.Parameters.priority,
                                            description="Waiting while {} get deployed".format(d_env.url),
                                            kill_timeout=120,
                                            environment=environment,
                                            tag=data['tag'],
                                            wait=True,
                                            deploy_with_robot=True,
                                            deployer_username=MBCONFIG.deployer_username)
            running_tasks.append(task)
        logging.debug('Processed yd environments result: %s', environment_map)
        if not running_tasks:
            return True
        return False

    def on_execute(self):
        super(MediabillingWatchDeployment, self).on_execute()
        # Sometimes this fields are None
        try:
            dashboards = json.loads(self.Parameters.dashboards)
        except ValueError:
            dashboards = {}
        try:
            yd_envs = json.loads(self.Parameters.yd_envs)
        except ValueError:
            yd_envs = {}

        logging.info('Processing dashboards: %s, yd_envs: %s', dashboards, yd_envs)
        secret = sdk2.yav.Secret(MBCONFIG.yav_token)
        token = secret.data()[MBCONFIG.token_name]

        nanny_ready = self._check_nanny(dashboards, token)
        logging.info('Nanny processed, result: %s (dashboards: %s)', nanny_ready, dashboards)
        yd_ready = self._check_yd(yd_envs)
        logging.info('YD processed, result: %s (yd_envs: %s)', yd_ready, yd_envs)
        all_ready = nanny_ready & yd_ready
        logging.info('All ready flag result: %s', all_ready)

        # check timeout
        running_seconds = (datetime.datetime.now(self.created.tzinfo) - self.created).total_seconds()
        message_id = None if self.Context.message_id is NotExists else self.Context.message_id
        if running_seconds > MBCONFIG.release_timeouts[self.Parameters.target]:
            on_duty_telegram_login = self.get_telegram_login(token, self.on_duty_backend(token))
            self.nyan_safe(self.template_timed_out(running_seconds, on_duty_telegram_login),
                           reply_to_message_id=message_id)
            self.report_task_failed(token, running_seconds)
            if self.Parameters.target == 'testing':
                self.deploy_tests()
            return

        # create / update the message
        if self.Parameters.target == 'testing':
            message = self.template_testing(self.dashboards_to_deployments(dashboards, yd_envs, 'deploy-testing'))
        elif self.Parameters.target == 'stable':
            message = self.template_stable(self.dashboards_to_deployments(dashboards, yd_envs, 'deploy-prod'))
        elif self.Parameters.target == 'hotfix':
            message = self.template_hotfix(self.dashboards_to_deployments(dashboards, yd_envs, 'deploy-prod'))
        else:
            raise errors.TaskFailure("Unknown target {}".format(self.Parameters.target))

        self.Context.message_id = self.nyan_safe(message, message_id)

        # check if ready
        if all_ready:
            if self.Parameters.target == 'testing':
                self.deploy_tests()
            self.nyan_safe(self.template_done(running_seconds),
                           reply_to_message_id=message_id)
            self.report_task_done(token)
            if self.Parameters.target == 'stable':
                self.set_issue_deployed_to_prod(token)
                startrek = StartrekHelper(token)
                arcadia = ArcadiaHelper(startrek)
                parsed_changelog, _, _, _ = arcadia.prepare_and_extract_latest_changelog(self.Parameters.branch)
                for login, tasks in parsed_changelog.iteritems():
                    for task in tasks:
                        self.summon_author_to_released_issue(startrek, task, login, self.Parameters.issue)
            return

        raise sdk2.WaitTime(100)
