# coding: utf-8

import logging
import random
import string
from datetime import datetime

from sandbox import sdk2
from sandbox.sdk2.helpers import process
from sandbox.projects.common import binary_task
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.common.vcs.arc import ArcCommandFailed
from sandbox.projects.release_machine.helpers import arcanum_helper

import sandbox.projects.mediabilling.deploy.resources as mres

arc_token_secret_key = "arc.token"
arcanum_token_secret_key = "arcanum"


class PlusArcanumApi(arcanum_helper.ArcanumApi):

    def get_review_requests(self, component, limit=100, offset=0, owner=None, open_only=True):
        query = self.build_query(open_only, owner, component)

        fields = ",".join(["id", "summary", "issues", "status", "updated_at", "created_at", "author"])

        response = self._do_get("v1/review-requests?limit={}&order=id&offset={}&fields=review_requests({})&query={}"
                                .format(limit, offset, fields, query))
        if not response:
            raise RuntimeError("Failed to get review-requests from arcanum")

        return response["data"]["review_requests"]

    @staticmethod
    def build_query(open_only, owner, component):
        query_parts = []
        if owner:
            query_parts.append("owner({})".format(owner))
        if open_only:
            query_parts.append("open()")
        query_parts.append("label({}/merge-to-testing)".format(component))
        logging.info("Arcanum API query param: %s", ";".join(query_parts))
        return ";".join(query_parts)

    def get_pr_v1(self, rr_id):
        fields = ",".join(["id", "vcs", "status"])
        return self._do_get("v1/pull-requests/{}?fields={}"
                            .format(rr_id, fields))


class PlusTestBranch(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        description = "Merges several pull requests into one branch"
        yav_secret = sdk2.parameters.String(u"Секрет с arc и arcanum токенами под ключами arc.token и arcanum. Не используется, если включен use_robot",
                                                        default="sec-01d8gvavayt5df53kwk3kfnzzn")
        component = sdk2.parameters.String("Компонента для сборки пров", default_value="default")
        prs_number = sdk2.parameters.Integer("Максимальное количество пров", default_value=10)
        create_pr = sdk2.parameters.Bool("Создать пр из новой ветки в транк", default=False)
        prerelease_branch = sdk2.parameters.Bool("Создавать ветку для предрелиза", default=True)
        prerelease_num = sdk2.parameters.Integer("Номер предрелиза", required=False)
        use_robot = sdk2.parameters.Bool("Использовать робота для работы с arc и arcanum. Секреты для робота берутся из sb vault", default=True)
        robot_name = sdk2.parameters.String("Робот для создания ветки", default_value="robot-afisha-deploy")
        with sdk2.parameters.Output:
            arcanum_review_id = sdk2.parameters.String("arc review id")
            branch = sdk2.parameters.String("result branch")

    @property
    def user(self):
        if not self.Parameters.use_robot:
            return self.author
        return self.Parameters.robot_name

    def _init_arc(self):
        if self.Parameters.use_robot:
            vault = "{}.arc-token".format(self.user)
            token = sdk2.Vault.data(self.user, vault).rstrip()
        else:
            secret = sdk2.yav.Secret(self.Parameters.yav_secret)
            token = secret.data()[arc_token_secret_key]
        self.arc = Arc(arc_oauth_token=token)

    def _init_arcanum(self):
        if self.Parameters.use_robot:
            vault = "{}.arcanum".format(self.user)
            token = sdk2.Vault.data(self.user, vault).rstrip()
        else:
            secret = sdk2.yav.Secret(self.Parameters.yav_secret)
            token = secret.data()[arcanum_token_secret_key]
        self.arcanum = PlusArcanumApi(token=token)

    def _arc_delete_branch(self, mount_point, branch_name):
        logging.info("Deleting branch {}".format(branch_name))
        try:
            command = [self.arc.binary_path, "push", "-d", branch_name]
            process.subprocess.check_call(command, cwd=mount_point)
        except process.subprocess.CalledProcessError as e:
            logging.info("nothing to delete. Error: %s", e)

    def get_prs(self):
        result = []
        page_size = 100
        page_number = 0

        while True:
            page = self.arcanum.get_review_requests(self.Parameters.component, limit=page_size, offset=page_number * page_size)
            for pr in page:
                result.append(pr)
                if len(result) == self.Parameters.prs_number:
                    return result
            if len(page) < page_size:
                return result
            page_number += 1

    def on_execute(self):
        super(PlusTestBranch, self).on_execute()
        self._init_arc()
        self._init_arcanum()

        branches = []
        prs_info = [{"id": pr["id"], "summary": pr["summary"], "issues": pr["issues"]} for pr in sorted(self.get_prs(), key=lambda pr: pr["created_at"])]
        self.Context.prs_info = prs_info
        pr_ids = [pr["id"] for pr in sorted(self.get_prs(), key=lambda pr: pr["created_at"])]
        id_author = {pr["id"]: pr["author"]["name"] for pr in sorted(self.get_prs(), key=lambda pr: pr["created_at"])}
        branch_author = dict()
        for pr_id in pr_ids:
            pr = self.arcanum.get_pr_v1(pr_id)["data"]
            logging.info("Pull request {} = {}", pr_id, pr)
            if self.is_pr_for_testing(pr):
                branch_author[pr["vcs"]["from_branch"]] = id_author[pr["id"]]
                branches.append(pr["vcs"]["from_branch"])

        if not branches:
            raise RuntimeError("Can't find any prs for %s pre-release" % self.Parameters.component)

        with self.arc.mount_path("", changeset="trunk", fetch_all=False) as mount_point:
            merged_prs_count = 0
            previous_branch = "trunk"
            for branch_name in branches:
                self.arc.checkout(mount_point=mount_point, branch=branch_name)
                try:
                    self.arc.rebase(mount_point=mount_point, upstream="trunk")
                except ArcCommandFailed as exc:
                    logging.info("Failed to rebase %s onto trunk: %s", branch_name, exc.args[0])
                    self.Context.merge_conflicts = True
                    self.Context.merge_condidates = (branch_name, "trunk")
                    self.Context.conflicts_authors = (branch_author[branch_name],)
                    self.rebase_abort(mount_point)
                    raise RuntimeError("Merge conflicts. More info in task logs")
                try:
                    self.arc.rebase(mount_point=mount_point, upstream=previous_branch)
                    previous_branch = branch_name
                    merged_prs_count += 1
                except ArcCommandFailed as exc:
                    logging.info("Failed to rebase %s onto %s: %s", branch_name, previous_branch, exc.args[0])
                    self.Context.merge_conflicts = True
                    self.Context.merge_condidates = (branch_name, previous_branch)
                    self.Context.conflicts_authors = (branch_author[branch_name], branch_author[previous_branch])
                    self.rebase_abort(mount_point)
                    raise RuntimeError("Merge conflicts. More info in task logs")

            if merged_prs_count:

                if self.Parameters.prerelease_branch:
                    result_branch_name = "prerelease_{}_{}".format(self.Parameters.component, self.Parameters.prerelease_num)
                    self._arc_delete_branch(mount_point, "users/{}/{}".format(self.user, result_branch_name))
                else:
                    result_branch_name = self.branch_name()
                self.arc.checkout(mount_point=mount_point, create_branch=True, branch=result_branch_name)
                upstream = "users/{}/{}".format(self.user, result_branch_name)
                self.arc.push(mount_point=mount_point,
                              upstream=upstream,
                              force=True)
                now = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
                message = "Testing branch with {} prs ({})".format(merged_prs_count, now)
                if self.Parameters.create_pr:
                    pr = self.arc.pr_create(mount_point=mount_point, message=message, publish=True)
                    # pr contents (multiline string):
                    #  Creating Arcanum PR
                    #  Local branch: testing-8kq3whi74d
                    #  Remote branch: users/robot-mediabilling/testing-8kq3whi74d
                    #  Arcanum PR is successfully created
                    #
                    #  https://a.yandex-team.ru/review/2204533
                    self.Parameters.arcanum_review_id = pr.split()[-1].split("/")[-1]
                self.Parameters.branch = upstream
                resource = mres.PlusTestBranchName(self, "Branch for testing", "branch_name.txt")
                resource.path.write_bytes(upstream)
            else:
                raise RuntimeError("Can't find any prs for %s pre-release", self.Parameters.component)

    def rebase_abort(self, source_root):
        command = [self.arc.binary_path, "rebase", "--abort"]
        rebase_abort_output = self.arc._execute_command(mount_point=source_root, command=command)
        logging.info(rebase_abort_output)

    @staticmethod
    def is_pr_for_testing(pr):
        return pr["vcs"]["type"] == "arc" and pr["vcs"]["to_branch"] == "trunk"

    @staticmethod
    def branch_name():
        alphabet = string.ascii_lowercase + string.digits
        digits = string.digits
        random.seed(datetime.now())
        return 'testing-' + ''.join(random.sample(alphabet, 10)) + '.' + ''.join(random.sample(digits, 10))
