# coding: utf-8

import locale
import logging
import os
import re
from collections import defaultdict
from datetime import datetime

import jinja2

from sandbox.sdk2.vcs.svn import Arcadia


class TaskInfo(object):
    def __init__(self):
        self.key = None
        self.qa_engineer = None
        self.status = None
        self.commits = []
        self.components = []

    @property
    def last_commit(self):
        last_date = datetime(1900, 1, 1)
        for commit in self.commits:
            if commit['date'] > last_date:
                last_date = commit['date']
        return last_date.strftime('%-d %b, %-H:%M').decode('utf-8').replace(u' ', u' ')  # nbsp


class ArcadiaHelper(object):
    _startrek_re = re.compile(r'\b([A-Z]+\-\d+)\b')
    _review_re = re.compile(r'\n\nREVIEW: (\d+).*', re.DOTALL)
    _release_branch_re = re.compile(r'^(stable-\d+)/$')
    _branch_dir = 'arcadia:/arc/branches/mediabilling/common'

    def __init__(self, startrek):
        from library.python import resource
        template = resource.find(os.path.dirname(__file__) + '/startrek.jinja').decode('utf-8')
        self._template = jinja2.Template(template)
        self._startrek = startrek

    @classmethod
    def get_latest_affected_revision(cls, arc_url):
        return cls.get_latest_affected_revisions(arc_url, 1)[0]

    @staticmethod
    def get_latest_affected_revisions(arc_url, limit=10):
        log = Arcadia.log(arc_url, 'HEAD', 'r0', limit=limit, stop_on_copy=True)
        return [x['revision'] for x in log]

    @classmethod
    def get_url(cls, branch, subdir=None):
        url = '{}/{}'.format(cls._branch_dir, branch.strip("/"))
        if subdir:
            url = '{}/{}'.format(url, subdir.strip("/"))
        return url

    def extract_msg_info(self, msg):
        match = re.search(self._review_re, msg)
        if match:
            review = match.group(1)
            msg = msg.replace(match.group(0), '')
        else:
            review = None

        tasks = re.findall(self._startrek_re, msg)
        for task in tasks:
            msg = msg.replace(task, '')
        if not tasks:
            tasks = []

        msg = msg.replace('\n', ' ')

        return review, tasks, msg.strip()

    def extract_changelog(self, from_revision, branch):
        result = defaultdict(lambda: defaultdict(TaskInfo))     # [login] -> [task] -> TaskInfo
        commits_without_task = []
        last_revision = from_revision
        for url in ['arcadia/media-billing']:
            for commit in Arcadia.log(self.get_url(branch, subdir=url), from_revision, 'HEAD'):
                review, tasks, msg = self.extract_msg_info(commit['msg'])
                author = commit['author']
                revision = commit['revision']
                last_revision = max(last_revision, revision)
                logging.info('Found commit %s from %s (%s): %s', revision, author, msg, tasks)

                for task in tasks:
                    st_info = self._startrek.get_info(task)

                    ti = result[author][task]
                    ti.key = task
                    ti.qa_engineer = st_info['qa_engineer']
                    ti.status = st_info['status']
                    ti.commits.append(commit)
                    ti.components = st_info['components']

                if not tasks:
                    commits_without_task.append({
                        'revision': revision,
                        'msg': msg,
                        'author': author,
                        'review': review,
                    })

        commits_without_task.sort(cmp=lambda a, b: cmp(a['author'], b['author']) or cmp(a['revision'], b['revision']))
        return result, commits_without_task, last_revision

    def extract_changelog_path(self, url, from_revision):
        result = defaultdict(lambda: defaultdict(TaskInfo))     # [login] -> [task] -> TaskInfo
        commits_without_task = []
        for commit in Arcadia.log(url, from_revision, 'HEAD'):
            review, tasks, msg = self.extract_msg_info(commit['msg'])
            author = commit['author']
            revision = commit['revision']

            logging.info('Found commit %s from %s (%s): %s', revision, author, msg, tasks)

            for task in tasks:
                st_info = self._startrek.get_info(task)

                ti = result[author][task]
                ti.key = task
                ti.qa_engineer = st_info['qa_engineer']
                ti.status = st_info['status']
                ti.commits.append(commit)
                ti.components = st_info['components']

            if not tasks:
                commits_without_task.append({
                    'revision': revision,
                    'msg': msg,
                    'author': author,
                    'review': review,
                })

        last_revision = commits_without_task[-1]['revision']
        commits_without_task.sort(cmp=lambda a, b: cmp(a['author'], b['author']) or cmp(a['revision'], b['revision']))
        return result, commits_without_task, last_revision

    def render_changelog(self, parsed_changelog, commits_without_tasks,
                         from_revision, to_revision, branch, on_duty, on_duty_testing, hotfix=False):
        loc = locale.getlocale()
        locale.setlocale(locale.LC_ALL, 'ru_RU.UTF-8')
        data = self._template.render(data=parsed_changelog,
                                     commits_without_tasks=commits_without_tasks,
                                     from_revision=from_revision,
                                     to_revision=to_revision,
                                     branch=branch,
                                     on_duty=on_duty,
                                     on_duty_testing=on_duty_testing,
                                     hotfix=hotfix)
        locale.setlocale(locale.LC_ALL, loc[0])
        return data

    def branch_to_revision(self, branch):
        path = self.get_url(branch)
        branch_revision = Arcadia.log(path, 'r0', 'HEAD', limit=1, stop_on_copy=True)[0]["revision"]
        last_commit_before = int(Arcadia.log(path, branch_revision, 'r0', limit=2)[-1]["revision"]) + 1
        return last_commit_before

    def find_previous_branch(self, branch):
        dirs = Arcadia.list(self._branch_dir, as_list=True)
        branches = []
        for dir_ in dirs:
            match = self._release_branch_re.match(dir_)
            if not match:
                continue
            branches.append(match.group(1))
        branches = sorted(branches, cmp=lambda a, b: cmp(int(a.split('-')[1]), int(b.split('-')[1])))
        return branches[branches.index(branch) - 1]

    def prepare_and_extract_latest_changelog(self, branch):
        previous_branch = self.find_previous_branch(branch)
        logging.info('Current branch: {}, previous: {}'.format(branch, previous_branch))
        first_revision = self.branch_to_revision(previous_branch)
        logging.info('Revisions: from {}'.format(first_revision))
        parsed_changelog, commits_without_task, last_revision = self.extract_changelog(first_revision, branch)
        return parsed_changelog, commits_without_task, first_revision, last_revision
