import logging
import os
import shutil
import subprocess
from collections import Callable
from datetime import datetime
from textwrap import dedent

import requests
import sandbox.projects.common.arcadia.sdk as asdk
import sandbox.projects.mediabilling.deploy.resources as mres
import sandbox.sdk2.vcs.svn as svn
from sandbox import sdk2
from sandbox.sdk2 import Task, Resource

from sandbox.projects.mediabilling.deploy.util import TaskHelper

from .ArcadiaHelper import ArcadiaHelper


# these are for pyflakes tests to pass
assert Task
assert Resource
assert Callable


class BuildHelper(object):
    _BUILD_DIR = 'build'
    _OUT_DIR = 'out'
    _BUILD_TARGETS = [
        'media-billing'
    ]
    _RETRIEVE_TARGETS = ['media-billing', 'media-billing/connectors']

    def __init__(self,
                 patch,                 # type: str
                 local_arcadia_path,    # type: str
                 set_info,              # type: Callable[[str]]
                 arcanum_token,         # type: str
                 ):
        self._patch = patch
        self._local_arcadia_path = local_arcadia_path
        self._set_info = set_info
        self._arcanum_token = arcanum_token

        self._branch = ''               # type: str
        self._revision = 0              # type: int

    def _patch_code(self,
                    path,               # type: str
                    ):

        if not self._patch:
            return

        patch = self._patch

        if patch.endswith('.zipatch'):
            self._set_info('Zipatching from {}'.format(patch))
            svn.Arcadia.apply_patch(path, 'zipatch:' + patch, '.')
            return

        filename = os.path.abspath('./own.arcadia.patch')
        self._set_info('Fetching {} into {}'.format(patch, filename))
        r = requests.get(patch, headers={'Authorization': 'OAuth {}'.format(self._arcanum_token)})
        r.raise_for_status()
        with open(filename, 'w') as fout:
            # temporary crutch, remove when https://a.yandex-team.ru/review/976255/files/1 gets merged in
            fout.write(str(r.text).replace('trunk/arcadia/', ''))

        logging.info(subprocess.check_output(['ls', '-l', filename]))
        self._set_info('Patching')
        svn.Arcadia.apply_patch_file(path, filename, False)

    def _copy_jars(self):
        out_dir = os.path.join(self._OUT_DIR, 'lib', 'jars')
        os.makedirs(out_dir)
        found = {}
        common = None
        common_name = 'common-mediabilling'
        os.makedirs(os.path.join(out_dir, common_name))
        for target in self._RETRIEVE_TARGETS:
            in_dir = os.path.join(self._BUILD_DIR, target)
            # find diffs
            for pack in os.listdir(in_dir):
                try:
                    files = os.listdir(os.path.join(in_dir, pack, pack))
                except OSError as x:
                    logging.error(x)
                    continue
                found[pack] = set(files)
                common = found[pack] if common is None else common.intersection(found[pack])
        common_pack = None
        for target in self._RETRIEVE_TARGETS:
            in_dir = os.path.join(self._BUILD_DIR, target)
            # copy diffs
            for pack in os.listdir(in_dir):
                if pack not in found:
                    continue
                common_pack = pack
                os.makedirs(os.path.join(out_dir, pack))
                for file_name in found[pack] - common:
                    shutil.copy(os.path.join(in_dir, pack, pack, file_name),
                                os.path.join(out_dir, pack, file_name))

        for file_name in common:
            shutil.copy(os.path.join(in_dir, common_pack, common_pack, file_name),
                        os.path.join(out_dir, common_name, file_name))

    def _build_code(self,
                    path,               # type: str
                    arcadia_url,        # type: str
                    yt_cache_token,     # type: str
                    ):
        self._set_info('Building {}'.format(arcadia_url))
        asdk.do_build(build_system=asdk.consts.YMAKE_BUILD_SYSTEM,
                      source_root=path,
                      targets=self._BUILD_TARGETS,
                      results_dir=self._BUILD_DIR,
                      clear_build=False,
                      def_flags={'JDK_VERSION': '11'},
                      yt_store_params=asdk.YtStoreParams(True, yt_cache_token))
        self._set_info('Build finished')

    def _copy_bootstrap(self,
                        path,  # type: str
                        ):
        shutil.copytree('{}/media-billing/bootstrap'.format(path),
                        os.path.join(self._OUT_DIR, 'bootstrap'))
        shutil.copy('{}/infra/rtc/packages/yandex-search-ip-tunnel/ip_tunnel.py'.format(path),
                    '{}/bootstrap/common/usr/bin/ip_tunnel.py'.format(self._OUT_DIR))

    def _generate_version_properties(self,
                                     url,       # type: str
                                     out_path,  # type: str
                                     ):
        date = datetime.now().strftime('%Y-%m-%d')
        time = datetime.now().strftime('%H:%M:%S')
        version_properties = dedent("""
                                    build.date={date}T{time}
                                    project.svn.url={url}
                                    project.svn.rev={revision}
                                    project.version={date}.{branch}.{revision}
                                    """.format(revision=self._revision,
                                               branch=self._branch,
                                               url=url,
                                               date=date, time=time))
        mediabilling_version_out_path = os.path.abspath(out_path + '/ru/yandex/mediabilling/mediabilling-version.properties')
        os.makedirs(os.path.dirname(mediabilling_version_out_path))
        with open(mediabilling_version_out_path, 'w') as vop:
            vop.write(version_properties)
        with open(out_path + '/mediabilling-version.app.properties', 'w') as omvp:
            omvp.write(version_properties)

    def _checksum(self):
        self._set_info('Checksumming result')
        sums = subprocess.check_output('find {} -type f -print0 | xargs -0 sha256sum'
                                       .format(self._OUT_DIR),
                                       shell=True)
        sums = sums.replace('  out/bootstrap', '  bootstrap/bootstrap')
        sums = sums.replace('  out/lib', '  mediabilling/lib')
        open('{}/checksums'.format(self._OUT_DIR), 'w').write(sums)

    def _package_single_resource(self,
                                 task,              # type: Task
                                 cls,               # type: Resource.__class__
                                 description,       # type: str
                                 path,              # type: str
                                 add_attrs=False,   # type: bool
                                 ):
        resource = cls(task,
                       description='Mediabilling JARs {}for revision {}/{}'.format(description,
                                                                                   self._branch,
                                                                                   self._revision),
                       path='{}/{}'.format(self._OUT_DIR, path),
                       arch=None,
                       ttl=14 if self._patch else 180)
        if add_attrs:
            resource.branch = str(self._branch)
            resource.revision = int(self._revision)
            if self._patch is not None:
                resource.patch = self._patch
        data = sdk2.ResourceData(resource)
        data.ready()

    def _package_resources(self,
                           task,  # type: Task
                           ):
        self._set_info('Packaging resources')
        self._package_single_resource(task, mres.MediabillingJarsBootstrapResource, 'bootstrap ', 'bootstrap')
        self._package_single_resource(task, mres.MediabillingJarsChecksumResource, 'checksum ', 'checksums')
        self._package_single_resource(task, mres.MediabillingJarsResource, '', 'lib', not self._patch)

    def _execute_path(self,
                      url,          # type: str
                      path,         # type: str
                      yt_token,     # type: str
                      ):
        """
        Does only part of the work that requires to keep the mounted-in arcadia
        """
        self._set_info('Building {} into {}'.format(os.path.abspath(self._BUILD_DIR), os.path.abspath(self._OUT_DIR)))
        self._patch_code(path)
        self._build_code(path, url, yt_token)
        self._copy_jars()
        self._copy_bootstrap(path)

        self._generate_version_properties(url, self._OUT_DIR + '/lib/jars/common')
        self._generate_version_properties(url, self._OUT_DIR + '/lib/jars/common-mediabilling')

    @property
    def branch(self):
        return self._branch

    @property
    def revision(self):
        return self._revision

    def run(self,
            task,                   # type: Task
            url,                    # type: str
            yt_token,               # type: str
            ):
        """
        Runs the build and packages the resources.
        In the process sets up the branch and revision properties on self.
        """
        if self._local_arcadia_path:
            self._set_info('Running in local arcadia path mode')
            self._branch, revision = TaskHelper.extract_branch_and_revision(url)
            self._revision = revision if revision else 11111
            self._execute_path(url, self._local_arcadia_path, yt_token)
            self._checksum()
            self._package_resources(task)
        else:
            self._branch, revision = TaskHelper.extract_branch_and_revision(url)
            self._revision = revision if revision else ArcadiaHelper.get_latest_affected_revision(url)

            with asdk.mount_arc_path(url) as path:
                self._execute_path(url, path, yt_token)
                self._checksum()
                self._package_resources(task)
