# coding: utf-8

import logging
from datetime import datetime
from urllib import urlencode

import requests
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import sandbox.sdk2 as sdk2
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi

from sandbox.projects.mediabilling.deploy.resources import MediabillingJarsResource
from .ArcadiaHelper import ArcadiaHelper
from .Config import MBCONFIG
from .TaskHelper import TaskHelper


class Deployment(object):

    def __init__(self):
        secret = sdk2.yav.Secret(MBCONFIG.yav_token)
        self._token = secret.data()[MBCONFIG.token_name]

        self._staff = StaffApi(self._token)
        self._startrek = None
        self._arcadia = None

    @property
    def startrek(self):
        if not self._startrek:
            self._startrek = StartrekHelper(self._token, MBCONFIG.queue)
        return self._startrek

    @property
    def arcadia(self):
        if not self._arcadia:
            self._arcadia = ArcadiaHelper(self.startrek)
        return self._arcadia

    @property
    def staff(self):
        return self._staff

    @staticmethod
    def get_ci_url(revision, branch):
        params = {
            'project_id': MBCONFIG.ci_project_id,
            'branch': branch,
            'revision': revision,
            'toolchain': MBCONFIG.ci_toolchain,
            'max_revisions': MBCONFIG.ci_max_revisions,
        }
        return 'https://ci.yandex-team.ru/api/v1.0/timeline/?' + urlencode(params)

    @staticmethod
    def find_build_jars_task(revision, branch, pr_id=None, patch=None):
        task_attributes = {
            "revision": revision,
            "branch": branch,
        }
        if pr_id is not None:
            task_attributes["pr_id"] = pr_id
        if patch is not None:
            task_attributes["patch"] = patch
        resource = sdk2.Resource.find(
            type=MediabillingJarsResource,
            status=ctr.State.READY,
            attrs=task_attributes
        ).first()
        return resource.task if resource else None

    @staticmethod
    def find_build_docker_task(task_type, revision, branch, targets):
        return sdk2.Task.find(
            task_type,
            status=(ctt.Status.SUCCESS),
            output_parameters={
                'revision': revision,
                'branch': branch,
            },
            input_parameters={
                'targets': targets
            },
            children=True
        ).first()

    @staticmethod
    def get_revision_fail_count(revision, branch):
        """ Determine revision fail count. Does not raise errors but returns None on errors """
        url = Deployment.get_ci_url(revision, branch)
        logging.info('Checking revision {}/{} fail count from {}'.format(branch, revision, url))
        result = requests.get(url)
        try:
            result.raise_for_status()
            return int(result.json()[-1][u'fails'])
        except requests.HTTPError:
            return None
        except IndexError:
            return None

    @staticmethod
    def get_trunk_fail_count():
        url = 'https://ci.yandex-team.ru/api/v1.0/tests?new_stats=1&limit=1&project_id=787&offset=0'
        result = requests.get(url)
        try:
            result.raise_for_status()
            return int(result.json()['tests_stats']['FAILED'])
        except requests.HTTPError:
            return 0

    def find_latest_affected_revision(self):
        return max(self.arcadia.get_latest_affected_revision(MBCONFIG.arcadia_trunk + '/music'),
                   self.arcadia.get_latest_affected_revision(MBCONFIG.arcadia_trunk + '/media-billing'))

    def close_released_issue_if_needed(self, now):
        if TaskHelper.is_hour_for_new_branch(now) and TaskHelper.is_working_day(now):
            deployed_issues = self.startrek.find_issues_in_state('deployed')
            for issue in deployed_issues:
                issue.transitions["closed"].execute()

    def create_new_release_if_needed(self, testing_issues, new_release_task):
        if testing_issues:
            logging.info('Skipping a new release creation because there are testing pending releases')
            return

        if self.startrek.find_nonclosed_issues():
            logging.info('Skipping a new release creation because there are nonclosed tickets')
            return

        revision = self.find_latest_affected_revision()
        if not revision:
            return

        logging.info('Enqueing a new release from revision {}'.format(revision))
        new_release_task(url=MBCONFIG.arcadia_trunk_revision(revision)).enqueue()

    def start_deploy_to_stable_if_required(self, testing_issues, deploy_to_stable_task):
        logging.info('Initiating deployment to stable')
        if testing_issues:
            logging.info('Skipping deploy to stable because there is an testing release issue')
            return

        ready_to_deploy = self.startrek.find_ready_to_deploy_issues()
        if len(ready_to_deploy) != 1:
            logging.info('Ready to deploy issues count: {}'.format(len(ready_to_deploy)))
            return

        issue = ready_to_deploy[0]

        match = self.startrek.summary_revision_and_branch(summary=issue.summary)
        if not match:
            logging.error('Bad issue {} summary: {}'.format(issue.key, issue.summary))
            return

        build_task_id = int(issue.releaseNotes)
        tasks = list(sdk2.Task.find(id=build_task_id, children=True).limit(1))
        if not tasks:
            logging.error('Cannot find task {}'.format(build_task_id))
            return

        task = tasks[0]
        if str(task.type) != MBCONFIG.build_jars_task_type:
            logging.error('Task {} type is {}, not {}'.format(task.id, task.type, MBCONFIG.build_jars_task_type))
            return

        task = deploy_to_stable_task(task=task)
        issue.transitions['releasing'].execute()
        task.enqueue()

    def update_testing_issue(self, issue, new_release_task, token):
        need_update = False

        match = self.startrek.summary_revision_and_branch(summary=issue.summary)
        if not match:
            logging.error('Bad issue {} summary: {}'.format(issue.key, issue.summary))
            return

        old_revision, branch = match
        parsed_changelog, commits_without_task, first_rev, last_rev = self.arcadia.prepare_and_extract_latest_changelog(
            branch)

        logging.info('Found last_rev = {}'.format(last_rev))
        logging.info('Found old_revision = {}'.format(old_revision))
        if last_rev == old_revision:
            logging.info('No need to update the issue, as last branch revision equals {}'.format(old_revision))
            return

        text = self.arcadia.render_changelog(parsed_changelog=parsed_changelog,
                                             commits_without_tasks=commits_without_task,
                                             from_revision=first_rev,
                                             to_revision=last_rev,
                                             branch=branch,
                                             on_duty=TaskHelper.on_duty_backend(token),
                                             on_duty_testing=TaskHelper.on_duty_testing(token))

        if issue.description != text:
            logging.info('description changed from {}  to {}'.format(issue.description, text))
            need_update = True

        if not need_update:
            logging.info('No need to update the issue description {}'.format(issue.key))
            return

        if issue.description.splitlines()[0] != text.splitlines()[0]:
            logging.info('Enqueing release updating from revision {}'.format(last_rev))
            new_release_task(url=MBCONFIG.arcadia_branch_revision(branch, last_rev)).enqueue()

        logging.info('The issue description {} needs to be updated'.format(issue.key))

        issue.update(description=text)
        logging.info('Issue {} updated'.format(issue.key))

    def cron(self, is_manual, new_release_task, deploy_to_stable_task, token):
        now = datetime.now()

        testing_issues = self.startrek.find_issues_in_state('open,testing')
        for issue in testing_issues:
            self.update_testing_issue(issue, new_release_task, token)

        self.close_released_issue_if_needed(now)
        self.create_new_release_if_needed(testing_issues, new_release_task)
        if is_manual or TaskHelper.is_everybody_at_work(now):
            self.start_deploy_to_stable_if_required(testing_issues, deploy_to_stable_task)
