# coding: utf-8
import logging
import re
from datetime import datetime

import requests
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.environments as environments
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.music.deployment.helpers.Nyan import nyan
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi

from .Config import MBCONFIG

status_finish = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK


def subtaskable(do_raise):
    def real_subtaskable(func):
        def subtask_list(parent, func_args):
            subtasks = sdk2.Task.find(parent=parent).limit(25)
            subtask_dict = dict((str(task.type), task) for task in subtasks if task.status != ctt.Status.DRAFT)
            return [subtask_dict.get(arg.upper(), None) for arg in func_args]

        def wrapper(self, *args, **kwargs):
            func_name = func.__name__
            func_args = [x for x in func.func_code.co_varnames if x.startswith(('mediabilling',
                                                                                'create_branch',
                                                                                'afisha_deploy'))]

            if func_args:
                with getattr(self.memoize_stage, func_name + '::wait')(commit_on_entrance=False):
                    wait = [task for task in subtask_list(self, func_args) if task]
                    if wait:
                        raise sdk2.WaitTask(wait, status_finish, wait_all=True)

            with getattr(self.memoize_stage, func_name + '::run')(commit_on_entrance=False):
                extra_args = subtask_list(self, func_args) if func_args else []
                if do_raise:
                    for arg in extra_args:
                        if arg and arg.status == ctt.Status.EXCEPTION:
                            raise errors.TaskError('Subtask {} (#{}) Exception has occurred'.format(arg.type, arg.id))
                        elif arg and arg.status not in ctt.Status.Group.SUCCEED:
                            raise errors.TaskFailure('Subtask {} (#{}) has failed'.format(arg.type, arg.id))
                args = args + tuple(extra_args)

                return func(self, *args, **kwargs)

        return wrapper

    return real_subtaskable


class TaskHelper(sdk2.Task):
    startrek_client_environment = environments.PipEnvironment("startrek_client", version="1.7.0", use_wheel=True)
    _url_re = re.compile(r'^arcadia:/arc/(?:trunk|branches/(mediabilling/common|music)/([^/]+))/arcadia(?:@(\d+))?$')

    def enqueue_subtask(self, task_class, **kwargs):
        """
        :param task_class: sdk2.Task
        :type self: sdk2.Task
        """
        from afisha.infra.libs.base.funcs import to_sandbox_name
        if not self.subtasks:
            self.Context.subtasks = {}
        task = task_class(self, notifications=self.Parameters.notifications, **kwargs)
        task.enqueue()
        task_name = to_sandbox_name(type(task).__name__)
        if task_name not in self.subtasks:
            self.subtasks[task_name] = []
        if task.id not in self.subtasks[task_name]:
            self.subtasks[task_name].append(task.id)
        return task

    def check_there_is_only_one_such_task(self):
        """
        :type self: sdk2.Task
        """
        tasks = list(sdk2.Task.find(task_type=self.type,
                                    status=ctt.Status.Group.EXECUTE + ctt.Status.Group.WAIT,
                                    children=True).limit(2))
        if len(tasks) > 1:
            raise errors.TaskFailure('Found running {} tasks: {}'.format(self.type,
                                                                         ", ".join(str(t.id) for t in tasks)))

    def nyan_safe(
        self,
        text,
        edit_message_id=None,
        chat_id=MBCONFIG.mediabilling_releases_chat_id,
        reply_to_message_id=None,
        parse_mode="markdown",
        raise_exception=False
    ):
        sent_by_part = "\n\nОтправлено таской " + MBCONFIG.markdown_sandbox(self.id)
        return nyan(
            text + sent_by_part,
            edit_message_id=edit_message_id,
            chat_id=chat_id,
            reply_to_message_id=reply_to_message_id,
            parse_mode=parse_mode,
            disable_notification=not TaskHelper.is_everybody_at_work(datetime.now()),
            raise_exception=raise_exception
        )

    @staticmethod
    def filter_tasks_by_param(tasks_ids, key, value):
        logging.debug("Searching %s parameter key with %s value in % task ids", key, value, tasks_ids)
        for task_id in tasks_ids:
            task_ = sdk2.Task[task_id]
            if getattr(task_.Parameters, key, None) == value:
                logging.debug("Found in %s", task_id)
                return task_
            logging.debug("Not found in %s", task_id)
        logging.debug("Not found in %s tasks", tasks_ids)
        return None

    @property
    def subtasks(self):
        if self.Context.subtasks is ctm.NotExists:
            return {}
        return self.Context.subtasks

    @staticmethod
    def combine_branch_and_revision_to_url(branch, revision):
        return 'arcadia:/arc/{}/arcadia@{}'.format(branch, revision)

    @staticmethod
    def extract_branch_and_revision(url):
        m = TaskHelper._url_re.match(url)
        if not m:
            raise errors.TaskFailure('The URL is in incompatible format')
        project_tmp, branch, revision = m.groups()
        if not branch:
            branch = 'trunk'
        return branch, revision

    @staticmethod
    def on_duty_backend(token):
        return TaskHelper.on_duty_backend_json(token)['login']

    @staticmethod
    def on_duty_testing(token):
        return TaskHelper.on_duty_testing_json(token)['login']

    @staticmethod
    def on_duty_backend_json(token):
        return TaskHelper.on_duty_abc(token, MBCONFIG.abc_schedule_slag_backend)

    @staticmethod
    def on_duty_testing_json(token):
        return TaskHelper.on_duty_abc(token, MBCONFIG.abc_schedule_slag_qa)

    @staticmethod
    def on_duty_abc(token, schedule_slag):
        on_duty_login = None
        for tries in range(5):
            try:
                r = requests.get(MBCONFIG.abc_duty_url.format(schedule_slag),
                                 headers={'Authorization': 'OAuth {}'.format(token)})
                r.raise_for_status()
                on_duty_login = r.json()[0]['person']
                break
            except requests.RequestException as err:
                logging.warning('Cannot determine person on ABC duty: %s', err)
        return on_duty_login

    @staticmethod
    def get_telegram_login(token, staff_login):
        staff_api = StaffApi(token)
        try:
            response = staff_api.get_person_profile_property(staff_login, 'accounts')
            tg_login = [
                account['value']
                for account in response['accounts']
                if account['type'] == 'telegram' and not account['private']
            ][0]
            return tg_login
        except Exception as exc:
            logging.warning('Failed to get telegram login for %s: %s', staff_login, exc)
            return None

    @staticmethod
    def preprocess_template(template):
        return '\n'.join([x.strip() for x in template.strip().split('\n')])

    def check_authorization(self, login, token, auth_config):
        if MBCONFIG.is_dev:
            logging.info('Allowing from dev host')
            return

        # fallback for logins
        if login in auth_config['logins']:
            logging.info('Allowing build access for {} (explicit login)'.format(login))
            return

        staff_api = StaffApi(token)
        staff_groups = set(staff_api.get_user_groups(login))

        # check exclusions
        for group in auth_config['exclude']:
            if group in staff_groups:
                raise RuntimeError('Author {} is in the excluded build group {}'.format(login, group))

        # check inclusions
        for group in auth_config['include']:
            if group in staff_groups:
                logging.info('Allowing build access for {} (group {})'.format(login, group))
                return

        raise errors.AuthorizationError('Author {} is not present in any of {} groups'
                                        .format(login, ", ".join(auth_config['include'])))

    @staticmethod
    def is_working_day(now):
        return MBCONFIG.working_days['from'] <= now.weekday() <= MBCONFIG.working_days['to']

    @staticmethod
    def is_working_hour(now):
        return MBCONFIG.working_hours['from'] <= now.hour <= MBCONFIG.working_hours['to']

    @staticmethod
    def is_hour_for_new_branch(now):
        return MBCONFIG.new_branch_hours['from'] <= now.hour <= MBCONFIG.new_branch_hours['to']

    @staticmethod
    def is_everybody_at_work(now):
        if not (TaskHelper.is_working_hour(now) and TaskHelper.is_working_day(now)):
            logging.info('Skipping action due to the time bounds')
            return False
        return True
