# coding: utf-8
import datetime
import logging

import dateutil
import sandbox.sdk2 as sdk2
from sandbox.common.types.task import Status
from sandbox.projects.common import task_env, binary_task
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.sdk2 import parameters

from sandbox.projects.mediabilling.deploy.util import TaskHelper

yql_query_template = u"""
USE hahn;
PRAGMA yt.PoolTrees = "physical";
PRAGMA yt.TentativePoolTrees = "cloud";

INSERT INTO `{tmp_table}` WITH TRUNCATE
SELECT
    `id_checkId`,
    `id_iterationType`,
    `id_suiteId`,
    `id_testId`,
    `id_toolchain`,
    `id_iterationNumber`,
    `id_taskId`,
    `id_partition`,
    `id_retryNumber`,
    `autocheckChunkId`,
    `branch`,
    `created`,
    `isRight`,
    `isStrongMode`,
    `links`,
    `metrics`,
    `name`,
    `oldTestId`,
    `path`,
    `processedBy`,
    `requirements`,
    `resultType`,
    `revisionNumber`,
    `snippet`,
    `status`,
    `storageChunkNumber`,
    `storageChunkType`,
    `subtestName`,
    `tags`,
    `testOutputs`,
    `uid`
FROM `home/ci/storage/stable/test_result/daily/{table_name}`
WHERE `path` like "{path_prefix}%"
AND status == "TS_FLAKY"
ORDER BY `path`, `name`, `subtestName`, `revisionNumber`, `created`
"""


class FlakyTestAudit(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 60 * 60
        description = "Detect flaky tests, update startrek tickets"
        path_prefix = parameters.String(
            'Find tests, starting with prefix',
            required=True,
            description="e.g. media-billing/"
        )
        st_queue = parameters.String(
            'Startrek queue for tickets',
            required=True,
        )
        st_token = parameters.YavSecretWithKey(
            'Startrek OAuth token',
            required=True,
        )
        yt_token = parameters.YavSecretWithKey(
            'YT OAuth token',
            required=True,
        )
        yql_token_vault_name = parameters.String(
            'YQL token name (for internal RUN_YQL_2)',
            required=True,
        )

    def init_yt(self):
        from yt.wrapper import YtClient
        self.yt_client = YtClient(proxy='hahn.yt.yandex.net', token=self.Parameters.yt_token.value())

    def init_st(self):
        self.st_client = StartrekHelper(self.Parameters.st_token.value(), queue=self.Parameters.st_queue)

    def on_execute(self):
        super(FlakyTestAudit, self).on_execute()
        self.init_st()
        self.init_yt()

        with self.memoize_stage.run_yql:
            self.Context.flaky_runs_table = self.yt_client.create_temp_table(prefix="flaky-tests-")
            self.run_yql(self.Context.flaky_runs_table)
        flaky_test_runs = self.yt_client.read_table(self.Context.flaky_runs_table, format='json')

        runs_by_test = {}
        for row in flaky_test_runs:
            runs_by_test.setdefault(self.build_test_name(row), []).append(row)

        for test_name, runs in runs_by_test.items():
            logging.info("Processing test {} with {} flaky runs".format(test_name, len(runs)))
            ticket = self.find_test_ticket(test_name)
            if not ticket:
                ticket = self.st_client._startrek.issues.create(
                    queue=self.Parameters.st_queue,
                    summary=test_name)
            else:
                if ticket.resolvedAt:
                    resolved_at = dateutil.parser.isoparse(ticket.resolvedAt)
                    last_flaky_run = datetime.datetime.fromtimestamp(runs[-1]["created"] / 1000)
                    # reopen ticket
                    if last_flaky_run.timestamp() > resolved_at.timestamp():
                        self.st_client.set_issue_status(ticket.key, "open")
                        pass

            st_message = "===Recent flaky runs===\n" + "\n\n".join([self.serialize_run(run) for run in runs])
            self.st_client.add_comment(ticket.key, st_message)

    def find_test_ticket(self, test_name):
        tickets = list(self.st_client._startrek.issues.find(query=self.build_st_query(test_name)))
        ticket = tickets[0] if tickets else None
        return ticket

    def build_st_query(self, test_name):
        query = "Queue: {} AND Summary: #\"{}\"".format(self.Parameters.st_queue, test_name)
        return query

    def serialize_run(self, run):
        return (
            u'**{date}** - (({link_url} {link_title}))\n'
            u'Links: {links}\n'
            u'History: {new_history}\n'
            u'<{{Error\n%%{snippet}%%}}>\n\n'
        ).format(
            date=datetime.datetime.fromtimestamp(run["created"] / 1000).strftime('%Y-%m-%d %H:%M:%S'),
            link_title=self.revision_title(run),
            link_url=self.check_link(run),
            snippet=run["snippet"],
            links=self.logs(run),
            new_history=self.new_history_link(run),
        )

    def run_yql(self, tmp_table):
        from sandbox.projects.yql.RunYQL2 import RunYQL2
        yesterday = datetime.datetime.now() - datetime.timedelta(days=1)
        yql_query = yql_query_template.format(
            tmp_table=tmp_table,
            table_name=yesterday.strftime('%Y-%m-%d'),
            path_prefix=self.Parameters.path_prefix,
        )
        subtask = RunYQL2(
            self,
            description=self.Parameters.description,
            owner=self.owner,
        )
        subtask.Parameters.query = yql_query
        subtask.Parameters.yql_token_vault_name = self.Parameters.yql_token_vault_name
        subtask.Parameters.trace_query = True
        subtask.Parameters.use_v1_syntax = True
        subtask.Parameters.publish_query = True
        subtask.save()
        raise sdk2.WaitTask(subtask.enqueue().id, Status.Group.FINISH + Status.Group.BREAK)

    def logs(self, run):
        result = []
        for name, urls in run["links"].items():
            for url in urls:
                result.append("(({url} {name}))".format(name=name, url=url))
        return ", ".join(result)

    def revision_title(self, run):
        return self.trunk_commit_link(run) if run["branch"] == "trunk" else self.pr_link(run)

    def check_link(self, run):
        return self.check_over_trunk_link(run) if run["branch"] == "trunk" else self.check_over_pr_link(run)

    def trunk_commit_link(self, run):
        return 'https://a.yandex-team.ru/arcadia/commit/{commit}'.format(commit=run["revisionNumber"])

    def pr_link(self, run):
        return 'https://a.yandex-team.ru/review/{pr_id}'.format(pr_id=run["branch"][3:])

    def check_over_trunk_link(self, run):
        return (
            'https://a.yandex-team.ru/arcadia/commit/{commit}?checkId={check_id}'
            '&dialogId=CiCard'
            '&filter=resultType%28RT_ALL%29%3BsuiteCategory%28CATEGORY_CHANGED%29%3BspecialCase%28SPECIAL_CASE_FLAKY%29'
            '&iterationType={iteration_type}'
            '&number=0'
            '&openedItems={suite_id}%3ART_TEST_SUITE_MEDIUM%2C{suite_id}%3A{test_id}%3ART_TEST_MEDIUM'
            '&snippetViewMode=word-wrap'
        ).format(commit=run["revisionNumber"],
                 check_id=run["id_checkId"],
                 iteration_type=run["id_iterationType"],
                 suite_id=run["id_suiteId"],
                 test_id=run["id_testId"],
                 )

    def check_over_pr_link(self, run):
        return (
            'https://a.yandex-team.ru/review/{pr_id}/details?checkId={check_id}'
            '&dialogId=CiCard'
            '&filter=resultType%28RT_ALL%29%3BsuiteCategory%28CATEGORY_CHANGED%29%3BspecialCase%28SPECIAL_CASE_FLAKY%29'
            '&iterationType={iteration_type}'
            '&number=0'
            '&openedItems={suite_id}%3ART_TEST_SUITE_MEDIUM%2C{suite_id}%3A{test_id}%3ART_TEST_MEDIUM'
            '&snippetViewMode=word-wrap'
        ).format(pr_id=run["branch"][3:],
                 check_id=run["id_checkId"],
                 iteration_type=run["id_iterationType"],
                 suite_id=run["id_suiteId"],
                 test_id=run["id_testId"],
                 )

    def new_history_link(self, run):
        return "https://a.yandex-team.ru/tests/{suite_id}/{test_id}".format(
            suite_id=run["id_suiteId"],
            test_id=run["id_testId"]
        )

    def build_test_name(self, row):
        return "[{}] {}::{}".format(row["path"], row["name"], row["subtestName"])
