import datetime
import json
import logging

from bson import json_util, binary

import custom_functions
import util


def convert(mongo_export_json_line, spec):
    try:
        bson = json_util.loads(mongo_export_json_line)
    except Exception as e:
        logging.error("Failed to parse as bson: " + mongo_export_json_line)
        raise e
    try:
        return json.dumps(__convert_obj(bson, spec))
    except Exception as e:
        logging.error("Failed to convert line: " + str(bson))
        raise e


def __convert_value(value, spec):
    target_type = spec["type"]

    functions = spec.get("convert", [])
    for function in functions:
        function_to_call = getattr(custom_functions, function["function"])
        kwargs = function.get("args", {})
        value = function_to_call(value, **kwargs)

    if target_type == "boolean":
        if not isinstance(value, bool):
            raise RuntimeError("Expected bool type for " + value + ", actual " + type(value))
        return value
    if target_type == "string":
        return __any_to_str(value)
    if target_type == "double":
        return float(value)
    if target_type in ["int8", "int16", "int32", "int64",
                       "uint8", "uint16", "uint32", "uint64"]:
        return __any_to_number(value)
    if target_type == "object":
        if "items" in spec:
            return __convert_obj(value, spec["items"])
        if "convert" not in spec:
            raise Exception("'object' must have at least one of 'items' or 'convert' parameters")
        if not isinstance(value, dict):
            raise Exception("value of type 'object' must be converted to `dict`, actual is " + type(value))
        return value
    if target_type == "array":
        if "items" in spec:
            return [__convert_value(item, spec["items"]) for item in value]
        if "convert" not in spec:
            raise Exception("'array' must have at least one of 'items' or 'convert' parameters")
        if not isinstance(value, list):
            raise Exception("value of type 'array' must be converted to `list`, actual is " + type(value))
        return value
    raise RuntimeError("Unknown type: " + target_type)


def __convert_obj(value, spec):
    result = {}
    for field_name, field_spec in spec.items():
        source_field_name = field_spec.get("from", field_name)
        source_field_value = deep_get(value, source_field_name.split("/"))
        if source_field_value is None:
            if field_spec.get("required"):
                raise RuntimeError("Required field not found: " + source_field_name)
            else:
                continue
        result[field_name] = __convert_value(source_field_value, field_spec)
    return result


def deep_get(dictionary, keys):
    current_value = dictionary
    for key in keys:
        if isinstance(current_value, dict) and key in current_value:
            current_value = current_value[key]
        else:
            return None
    return current_value


def __any_to_number(dt):
    if isinstance(dt, datetime.datetime):
        return util.unix_time_millis(dt)
    return int(dt)


def __any_to_str(value):
    if isinstance(value, datetime.datetime):
        return value.isoformat()
    if isinstance(value, binary.Binary):
        return str(value.as_uuid())
    return str(value)
