import re
import logging
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common import errors
from sandbox.projects.common import binary_task
from sandbox.projects.common.nanny.client import NannyClient


NANNY_API_URL = "https://nanny.yandex-team.ru"
MAX_ITERATIONS = 3


class MediabillingNannyClient(NannyClient):

    def get_history_runtime_attrs_infos(self, service, limit=10, skip=0):
        url = '{}/v2/services/{}/history/runtime_attrs_infos/'.format(self._api_url, service)
        params = {
            "limit": limit,
            "skip": skip
        }
        return self._get_url(url, params=params)


class NannyChange(object):

    def __init__(self, service_id, ctime, author, comment, **kwargs):
        self.service_id = service_id
        self.ctime = ctime
        self.author = author
        self.comment = comment
        self.release_issue = self._find_release_issue(self.comment)
        if kwargs:
            logging.warning("Unknown attrs, ignored: %s", kwargs)

    @classmethod
    def from_dict(cls, data):
        logging.debug("Init NannyChange from data: %s", data)
        service_id = data["service_id"]
        change_info = data["change_info"]
        return cls(service_id, **change_info)

    @staticmethod
    def _find_release_issue(comment):
        pattern = re.compile(r"Update to .* \(task id (\d+)\)")
        try:
            task_id = pattern.search(comment).group(1)
        except AttributeError:
            return None
        try:
            release_task = sdk2.Task[task_id].parent.parent
            issue = release_task.Parameters.issue
        except AttributeError:
            return None
        return issue

    def to_dict(self):
        return {
            "service_id": self.service_id,
            "ctime": self.ctime,
            "author": self.author,
            "comment": self.comment,
            "release_issue": self.release_issue
        }

    def to_yt(self):
        result = self.to_dict()
        result["ctime"] = result["ctime"] * 1000  # Convert to microseconds for yt timestamp format (nanny returns unixtimestamp with ms)
        return result


class MediabillingReleaseLog(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            applications = sdk2.parameters.List("Applications to watch", required=True)
            secret = sdk2.parameters.YavSecret("Yav secret id", required=True)
            yt_path = sdk2.parameters.String("Path to yt table", default=None)

    def on_enqueue(self):
        mb_release_log_lock = ctt.Semaphores.Acquire(name="mb_release_log", weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[mb_release_log_lock], release=release)

    def _init_nc(self):
        return MediabillingNannyClient(NANNY_API_URL, self.secrets["NANNY_API_TOKEN"])

    def _init_yc(self):
        from yt.wrapper import YtClient
        yt_config = {
            "proxy": {
                "url": "hahn.yt.yandex.net"
            },
            "token": self.secrets["YT_API_TOKEN"]
        }
        return YtClient(config=yt_config)

    @staticmethod
    def _get_last_processed():
        last_task = sdk2.Task.find(task_type=MediabillingReleaseLog, status=ctt.Status.SUCCESS).first()
        if not last_task:
            return {}
        return last_task.Context.processed

    @property
    def secrets(self):
        if not getattr(self, "_secrets", None):
            self._secrets = self.Parameters.secret.data()
        return self._secrets

    @property
    def nc(self):
        if not getattr(self, "_nc", None):
            self._nc = self._init_nc()
        return self._nc

    @property
    def yc(self):
        if not getattr(self, "_yt", None):
            self._yc = self._init_yc()
        return self._yc

    @property
    def applications(self):
        return self.Parameters.applications

    @property
    def last_processed(self):
        if self.Context.last_processed is ctm.NotExists:
            self.Context.last_processed = self._get_last_processed()
        return self.Context.last_processed if self.Context.last_processed else {}

    @property
    def processed(self):
        if self.Context.processed is ctm.NotExists:
            self.Context.processed = {}
        return self.Context.processed

    @property
    def yt_path(self):
        return self.Parameters.yt_path

    def get_last_changes(self, application, last_ts, limit=10, skip=0, iteration=0):
        logging.info("Getting last changes for %s with params: last_ts=%s, limit=%s, skip=%s, iteration=%s", application, last_ts, limit, skip, iteration)
        result = []
        found = False
        if iteration > MAX_ITERATIONS:
            raise errors.TaskFailure("Smthng wrong, num of iterations excceded (%s/%s)" % (iteration, MAX_ITERATIONS))
        changes = self.nc.get_history_runtime_attrs_infos(application)["result"]
        for change in changes:
            change = NannyChange.from_dict(change)
            if change.ctime > last_ts:
                logging.info("change ctime > last_ts (%s > %s), appending to result", change.ctime, last_ts)
                result.append(change)
            else:
                logging.info("Found last processed ts in api, stopping search")
                found = True
                break
        if not found and last_ts != 0:
            logging.info("Still not found last processed ts in api and last_ts != 0 => continue search")
            result.extend(self.get_last_changes(application, last_ts, limit=limit, skip=skip+limit, iteration=iteration+1))
        return result

    def get_changes(self):
        changes = []
        for application in self.applications:
            logging.info("Processing %s", application)
            last_ts = self.last_processed.get(application, 0)
            logging.info("Got last ts: %s", last_ts)
            result = self.get_last_changes(application, last_ts)
            processed_ts = last_ts if not result else result[0].ctime
            logging.info("Saving %s ctime as last processed for %s", processed_ts, application)
            self.processed[application] = processed_ts
            changes.extend(result)
        changes.sort(key=lambda x: x.ctime)
        return changes

    def save_changes(self, changes):
        from yt.wrapper import TablePath
        if not self.yt_path or not changes:
            logging.info("yt_path not defined or no changes, skip save")
            return
        table = TablePath(self.yt_path, append=True)
        logging.info("Saving changes into %s table", self.yt_path)
        self.yc.write_table(table, [x.to_yt() for x in changes])

    def on_execute(self):
        super(MediabillingReleaseLog, self).on_execute()
        changes = self.get_changes()
        self.save_changes(changes)
