import logging
import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common import binary_task


class MediabillingYt2YtReplicate(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            source_cluster = sdk2.parameters.String("Source YT cluster", default="hahn", required=True)
            source_path = sdk2.parameters.String("Path to table, e.g.: //home/mediabilling/infra/sox/releases_log", required=True)
            dest_cluster = sdk2.parameters.String("Dest YT cluster", default="arnold", required=True)
            dest_path = sdk2.parameters.String("Path to table, e.g.: //home/mediabilling/infra/sox/releases_log", required=True)
            secret = sdk2.parameters.YavSecret("Yav secret id", required=True)
            secret_key = sdk2.parameters.String("Token key in secret from yav, default: YT_API_TOKEN", default="YT_API_TOKEN")

    def on_enqueue(self):
        lock_name = "mb_yt2yt_replicate_{}_{}_{}_{}".format(self.scluster, self.dcluster, self.spath, self.dpath)
        mb_yt2yt_replicate_lock = ctt.Semaphores.Acquire(name=lock_name, weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[mb_yt2yt_replicate_lock], release=release)

    @property
    def secrets(self):
        if not getattr(self, "_secrets", None):
            self._secrets = self.Parameters.secret.data()
        return self._secrets

    @property
    def token(self):
        return self.secrets[self.Parameters.secret_key]

    @property
    def scluster(self):
        return self.Parameters.source_cluster

    @property
    def dcluster(self):
        return self.Parameters.dest_cluster

    @property
    def spath(self):
        return self.Parameters.source_path

    @property
    def dpath(self):
        return self.Parameters.dest_path

    def _init_tmc(self):
        from yt.transfer_manager.client import TransferManager
        return TransferManager(token=self.token)

    @property
    def tmc(self):
        if not getattr(self, "_tmc", None):
            self._tmc = self._init_tmc()
        return self._tmc

    def on_execute(self):
        super(MediabillingYt2YtReplicate, self).on_execute()
        logging.info("Copying: %s:%s -> %s:%s", self.scluster, self.spath, self.dcluster, self.dpath)
        task_id = self.tmc.add_task(self.scluster, self.spath, self.dcluster, self.dpath)
        logging.info("Task: %s", task_id)
