# -*- coding: utf-8 -*-
import logging
import shutil

from sandbox import sdk2
from sandbox.common import errors
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.common.build.KosherYaMake import KosherYaMake
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
import sandbox.projects.websearch.begemot.resources as bg_resources


class BuildBeggins2(ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        ram = 10 * 1024
        disk_space = 30 * 1024

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3 * 60 * 60

        checkout_arcadia_from_url = sdk2.parameters.String(
            "Checkout url for arcadia",
            required=True,
        )

        ext_params = binary_task.LastBinaryReleaseParameters(stable=True)

    def on_save(self):
        super(BuildBeggins2, self).on_save()

    def on_execute(self):
        super(BuildBeggins2, self).on_execute()

        with self.memoize_stage.enqueue_and_wait_tasks():
            task_class = sdk2.Task["RELEASE_BEGEMOT_BEGGINS_RESOURCES"]
            build_resources_subtask = task_class(
                self,
                owner=self.Parameters.owner,
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                begemot_resources="BegemotConfig Bstr NannyConfigs Beggins BegginsExecutableAddons",
            ).enqueue()
            self.Context.build_resources_subtask_id = build_resources_subtask.id

            build_executable_subtask = KosherYaMake(
                self,
                owner=self.Parameters.owner,
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=True,
                aapi_fallback=True,
                build_system='semi_distbuild',
                build_type='release',
                result_single_file=True,
                check_return_code=True,
                failed_tests_cause_error=True,
                checkout=False,
                checkout_mode=False,
                clear_build=False,
                definition_flags='-DCUDA_VERSION=11.4 -DCUDNN_VERSION=8.0.5',
                result_rt='BEGEMOT_BEGGINS_EXECUTABLE',
                targets='search/daemons/begemot/beggins',
                arts='search/daemons/begemot/beggins/beggins',
                result_rd='Arcadia Project',
            ).enqueue()
            self.Context.build_executable_subtask_id = build_executable_subtask.id

            if 1 + 1 == 2:
                raise sdk2.WaitTask(
                    [
                        build_executable_subtask,
                        build_resources_subtask,
                    ],
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True,
                    timeout=60 * 60 * 3,
                )

        build_resources_subtask = self.find(id=self.Context.build_resources_subtask_id).first()
        build_executable_subtask = self.find(id=self.Context.build_executable_subtask_id).first()
        if build_resources_subtask.status != ctt.Status.SUCCESS:
            assert False, 'build resources subtask failed'

        if build_executable_subtask.status != ctt.Status.SUCCESS:
            raise errors.TaskFailure("Build executable subtask failed: {}".format(build_executable_subtask.status))

        self._copy_resources(build_executable_subtask, [bg_resources.BEGEMOT_BEGGINS_EXECUTABLE])
        self._copy_resources(
            build_resources_subtask,
            [
                bg_resources.BEGEMOT_ARGUMENTS_PARSER,
                bg_resources.BEGEMOT_BSTR,
                bg_resources.BEGEMOT_BSTR_CALLER,
                bg_resources.BEGEMOT_EVLOG_UPLOADER,
                bg_resources.BEGEMOT_EVLOGDUMP,
                "BEGEMOT_FAST_BUILD_CONFIG_BEGGINS",
                bg_resources.BEGEMOT_FAST_BUILD_DOWNLOADER,
                bg_resources.BEGEMOT_FAST_DATA_CALLBACK,
                bg_resources.BEGEMOT_INSTANCECTL_CONF,
                bg_resources.BEGEMOT_CONFIG,
            ],
        )

    def on_release(self, additional_parameters):
        ReleaseToNannyTask2.on_release(self, additional_parameters)

    def _copy_resources(self, subtask, target_resources_types):
        for target_resource_type in target_resources_types:
            child_resource = sdk2.Resource.find(task=subtask, type=target_resource_type).first()
            self._copy_resource(child_resource)
            logging.debug(
                'Copied data from {resource_type} resource of {id} task'.format(
                    resource_type=child_resource.type, id=subtask.id
                )
            )

    def _copy_resource(self, src_resource):
        if src_resource.state != ctr.State.READY:
            raise errors.TaskFailure("Child task resource is not ready, resource_id: {}".format(src_resource.id))
        src_path = sdk2.ResourceData(src_resource).path
        dst_resource = sdk2.Resource[src_resource.type](self, src_resource.description, src_resource.type, ttl='inf')
        dst_resource_data = sdk2.ResourceData(dst_resource)
        if src_path.is_dir():
            shutil.copytree(str(src_path), str(dst_resource_data.path))
        else:
            shutil.copy(str(src_path), str(dst_resource_data.path))
        dst_resource_data.ready()
