from sandbox import common
from sandbox import sdk2

import json
import logging
import os
import urllib
import re


ARCADIA_URL = 'svn+ssh://arcadia.yandex.ru/'
FORMULAS_DEFAULT_URL = os.path.join(ARCADIA_URL, 'robots/trunk/alice/megamind')
FORMULAS_PATH = 'megamind_formulas'

FML_DEFAULT_URL = 'https://fml.yandex-team.ru/download/computed/formula' \
                  '?id={}&file=matrixnet.{}'

DEFAULT_SOURCE_FORMULA_EXTENSION = 'info'
DEFAULT_TARGET_FORMULA_EXTENSION = 'info'

ALICE_URL = os.path.join(ARCADIA_URL, 'arc/trunk/arcadia/alice')
DEFAULT_PACKAGE_URLS = (
    os.path.join(ALICE_URL, 'vins/packages/vins_package.json'),
    os.path.join(ALICE_URL, 'megamind/deploy/packages/megamind_standalone.json'),
)
DEFAULT_CLASSIFICATION_CONFIG_URL = os.path.join(ALICE_URL, 'megamind/configs/common/classification.pb.txt')


def patch_package_description(path, id):
    with open(path, 'r') as file:
        description = json.load(file)

    modified = False
    for entry in description.get('data', []):
        if entry.get('destination', {}).get('path', '').endswith('/formulas'):
            entry['source'] = {
                'type': 'SANDBOX_RESOURCE',
                'id': id
            }
            modified = True
            break
    if not modified:
        raise common.errors.TaskError(
            "Can't find formulas section in the package description")

    with open(path, 'w') as file:
        json.dump(description, file, indent=4, sort_keys=True, separators=(',', ': '))


def patch_classification_config(path, config_patch):
    import google.protobuf.text_format as text
    import alice.megamind.library.config.protos.classification_config_pb2 as pb_cnf
    with open(path, 'r') as f:
        config = text.Parse(f.read(), pb_cnf.TClassificationConfig())
    if config_patch:
        for scenario_name in config.ScenarioClassificationConfigs:
            scenario_config = config.ScenarioClassificationConfigs[scenario_name]
            logging.info('looking at scenario: {}'.format(scenario_name))
            if scenario_name in config_patch.ScenarioClassificationConfigs:
                scenario_config_patch = config_patch.ScenarioClassificationConfigs[scenario_name]
                logging.info('Found')
                for formula_description in scenario_config_patch.FormulasDescriptionList.FormulasDescription:
                    new_formula_description = scenario_config.FormulasDescriptionList.FormulasDescription.add()
                    new_formula_description.MergeFrom(formula_description)
    with open(path, 'w') as f:
        f.write(text.MessageToString(config, as_utf8=True, use_short_repeated_primitives=True))


def write_config_to_resource(formulas_local_path, message):
    import google.protobuf.text_format as text
    with open(os.path.join(formulas_local_path, 'formulas_config.pb.txt'), 'a') as config:
        config.write(text.MessageToString(message, as_utf8=True))


def add_custom_formulas(custom_formulas, formulas_local_path, source_formula_extension, target_formula_extension):
    import alice.megamind.library.classifiers.formulas.protos.formulas_description_pb2 as pb_fml
    import alice.megamind.library.config.protos.classification_config_pb2 as pb_cnf
    import alice.megamind.protos.quality_storage.storage_pb2 as pb_strg
    import alice.protos.data.language.language_pb2 as pb_lang

    message = pb_cnf.TClassificationConfig()

    message.DefaultScenarioClassificationConfig.UseFormulasForRanking = False
    message.DefaultScenarioClassificationConfig.PreclassifierConfidentScenarioThreshold = 1.0

    new_experiment_dir = None
    for data_string in custom_formulas:
        data = json.loads(data_string)
        if "ScenarioName" not in data:
            raise common.errors.TaskError(
            "Formulas description doesn't contain \"ScenarioName\"")
        if "ClassificationStage" not in data:
            raise common.errors.TaskError(
            "Formulas description doesn't contain \"ClassificationStage\"")
        if "ClientType" not in data:
            raise common.errors.TaskError(
            "Formulas description doesn't contain \"ClientType\"")
        if "FormulaID" not in data:
            raise common.errors.TaskError(
            "Formulas description doesn't contain \"FormulaID\"")

        if not new_experiment_dir:
            new_dir_prefix = data.get("Experiment", "new_learn")
            client_path = os.path.join(formulas_local_path, data["ClientType"])
            if not os.path.exists(client_path):
                os.makedirs(client_path)
            matches = [re.match(new_dir_prefix + r'_([0-9]*)', f) for f in os.listdir(client_path)]
            new_num = str(max([0] + [int(x.group(1)) if x else 0 for x in matches]) + 1)
            if len(new_num) == 1:
                new_num = '0' + new_num
            new_experiment_dir = os.path.join(client_path, new_dir_prefix + '_' + new_num)

        scenario_config = message.ScenarioClassificationConfigs[data["ScenarioName"]]

        scenario_config.UseFormulasForRanking = True

        formula_description = scenario_config.FormulasDescriptionList.FormulasDescription.add()
        formula_description.Key.ClassificationStage = pb_strg.EMmClassificationStage.Value(data["ClassificationStage"])
        formula_description.Key.ClientType = pb_fml.EClientType.Value(data["ClientType"])
        if "Experiment" in data:
            formula_description.Key.Experiment = data["Experiment"]

        if "Language" in data:
            formula_description.Key.Language = pb_lang.ELang.Value(data["Language"])
        else:
            formula_description.Key.Language = pb_lang.ELang.L_RUS

        formula_name = "{}.{}".format(data["FormulaID"], data["ScenarioName"])
        formula_description.FormulaName = formula_name
        if "Threshold" in data:
            formula_description.Threshold.value = data["Threshold"]
        if "ConfidentThreshold" in data:
            formula_description.ConfidentThreshold.value = data["ConfidentThreshold"]

        full_path = os.path.join(new_experiment_dir, formula_name + '.' + target_formula_extension)
        if not os.path.exists(os.path.dirname(full_path)):
            os.makedirs(os.path.dirname(full_path))

        urllib.urlretrieve(
            FML_DEFAULT_URL.format(data["FormulaID"], source_formula_extension),
            full_path)

    return new_experiment_dir, message


def svn_update(path):
    path = os.path.normpath(path)
    parts = path.split('/')
    for i in range(2, len(parts)):
        sdk2.vcs.svn.Arcadia.update(os.path.join(*parts[:i]), depth='empty')
    sdk2.vcs.svn.Arcadia.update(path)


class FormulasResource(sdk2.Resource):
    """Resource with MM formulas"""
    ttl = 'inf'


class ReleaseMegamindFormulas(sdk2.Task):
    """Updates VINS+MM build specification with latest formulas"""

    class Parameters(sdk2.Task.Parameters):
        formulas_url = sdk2.parameters.String(
            'Arcadia URL to formulas',
            default=FORMULAS_DEFAULT_URL)
        formulas_local_path = sdk2.parameters.String(
            'Local path to formulas',
            default=FORMULAS_PATH)
        formulas_revision = sdk2.parameters.String(
            'Formulas revision',
            default=None)
        package_urls = sdk2.parameters.List(
            'Arcadia URL to packages',
            default=DEFAULT_PACKAGE_URLS)
        config_url = sdk2.parameters.String(
            'Arcadia URL to classification config',
            default=DEFAULT_CLASSIFICATION_CONFIG_URL)
        custom_formulas = sdk2.parameters.List(
            'Custom formulas (formula + FML id)',
            description='Formula has to be JSON-format string. Required fields: ScenarioName, ClassificationStage, ClientType, FormulaID. \
                Optional fields: Experiment, Threshold, ConfidentThreshold. Formula path will be megamind_formulas/experiments/<FormulaID>.info \
                Classifications Stages: ECS_POST, ECS_PRE. Current scenarios: Vins, HollywoodMusic, Video, Search',
            default={})
        source_formula_extension = sdk2.parameters.String(
            'Extension to download from fml',
            default=DEFAULT_SOURCE_FORMULA_EXTENSION)
        target_formula_extension = sdk2.parameters.String(
            'Extension for downloaded formulas',
            default=DEFAULT_TARGET_FORMULA_EXTENSION)
        commit_formulas = sdk2.parameters.Bool('Commit formulas binaries to robots', default=False)

        commit_configs = sdk2.parameters.Bool('Create config patch for MM', default=False)

        UseLastBinary = sdk2.parameters.Bool('Use last binary archive', default=True)
        with UseLastBinary.value[True]:
            with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                ReleaseType.values.stable = ReleaseType.Value('stable', default=True)
                ReleaseType.values.test = ReleaseType.Value('test')

        ssh_key_owner = sdk2.parameters.String('Ssh key owner', default='BASS')
        ssh_key_name = sdk2.parameters.String('Ssh key name', default='robot-bassist_ssh_key')
        commit_user = sdk2.parameters.String('Commit user', default='robot-bassist')
        commit_message = sdk2.parameters.String('Commit message', default='')

    def on_save(self):
        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'target': 'megamind/release_mm_formulas/bin', 'release': self.Parameters.ReleaseType or 'stable'}).first().id
        else:
            self.Requirements.tasks_resource = None

    def on_execute(self):
        formulas_url = self.Parameters.formulas_url
        formulas_local_path = self.Parameters.formulas_local_path
        formulas_revision = self.Parameters.formulas_revision
        package_urls = self.Parameters.package_urls
        config_url = self.Parameters.config_url
        custom_formulas = self.Parameters.custom_formulas
        commit_configs = self.Parameters.commit_configs
        source_formula_extension = self.Parameters.source_formula_extension
        target_formula_extension = self.Parameters.target_formula_extension
        commit_formulas = self.Parameters.commit_formulas
        ssh_key_owner = self.Parameters.ssh_key_owner
        ssh_key_name = self.Parameters.ssh_key_name
        commit_user = self.Parameters.commit_user
        commit_message = self.Parameters.commit_message
        if commit_message:
            if commit_message.endswith('.'):
                commit_message = commit_message[:-1]
            commit_message = commit_message + '. '

        sdk2.vcs.svn.Arcadia.checkout(
            url=formulas_url,
            path=formulas_local_path,
            revision=formulas_revision)

        info = sdk2.vcs.svn.Arcadia.info(formulas_local_path)
        logging.info('Checked out formulas: {}'.format(info))

        if custom_formulas:
            new_exp_path, config_update = add_custom_formulas(custom_formulas, formulas_local_path, source_formula_extension, target_formula_extension)
            logging.info('New formulas path: ' + new_exp_path)
            if commit_formulas:
                message = self.Parameters.commit_message + common.utils.get_task_link(sdk2.Task.current.id)
                message = '{}Sandbox task id: {}'.format(
                    commit_message,
                    self.id)
                with sdk2.ssh.Key(self, key_owner=ssh_key_owner, key_name=ssh_key_name):
                    try:
                        sdk2.vcs.svn.Svn.add(path=new_exp_path)
                    except sdk2.vcs.svn.SvnError:
                        # new_exp_path = client_path/experiment_path. Error means client_path is still not versioned
                        sdk2.vcs.svn.Svn.add(path=os.path.dirname(new_exp_path))
                    sdk2.vcs.svn.Svn.commit(path=formulas_local_path, message=message)
        else:
            config_update = None

        if not commit_configs:
            write_config_to_resource(formulas_local_path, config_update)

        resource = FormulasResource(
            task=self,
            description='Megamind formulas, revision {}'.format(info['commit_revision']),
            path=formulas_local_path,
            ttl='inf')
        sdk2.ResourceData(resource).ready()

        logging.info('Formulas resource id: {}'.format(resource.id))

        if not commit_configs:
            logging.info('Skip commit to megamind configs')
            return

        folder_url = os.path.commonprefix(package_urls + [config_url])
        while folder_url.endswith('/'):
            folder_url = folder_url[:-1]
        arc_url, folder = os.path.split(folder_url)
        sdk2.vcs.svn.Arcadia.checkout(folder_url, path=folder, depth='empty')
        for package_url in package_urls:
            package_path = os.path.relpath(package_url, arc_url)
            svn_update(package_path)
            patch_package_description(path=package_path, id=resource.id)
        config_path = os.path.relpath(config_url, arc_url)
        svn_update(config_path)
        patch_classification_config(config_path, config_update)

        message = '{}Sandbox task id: {}. Resource id: {} REVIEW:NEW'.format(
            commit_message,
            self.id,
            resource.id)
        with sdk2.ssh.Key(self, key_owner=ssh_key_owner, key_name=ssh_key_name):
            try:
                stdout = sdk2.vcs.svn.Arcadia.commit(path=folder, message=message, user=commit_user, with_revprop=['arcanum:review-publish=yes'])
                logging.info(stdout)
            except sdk2.vcs.svn.SvnError:
                pass  # Best code practices here
