import logging
import requests
import os
from sandbox import common
from sandbox import sdk2
import google.protobuf.text_format as text_format
from sandbox.sandboxsdk import svn
from datetime import datetime, timedelta
from sandbox.projects.common import binary_task


HAMSTER_VERSION_URL = 'http://vins.hamster.alice.yandex.net/version'
CONFIGS_FOLDER_NAME = 'current_production_scenario_configs'
SOLOMON_DATA_URL = 'https://solomon.yandex.net/api/v2/projects/megamind/sensors/data'
TOP_SCENARIOS_COUNT = 10
STAFF_LOGIN_TO_TG_URL = 'https://staff-api.yandex-team.ru/v3/persons?_fields=telegram_accounts&login='
ABC_SERVICE_URL = 'https://abc.yandex-team.ru/services/'
SOLOMON_ALERTS_URL = 'https://solomon.yandex.net/api/v2/projects/megamind/alerts/'
ABC_SLUG_URL = 'https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service__slug='
JUGGLER_CREATE_OR_UPDATE_CHECK_URL = 'https://juggler-api.search.yandex.net/api/checks/add_or_update?do=1'
SERVICE_PREFIX = 'scenario_errors_generated_alert_'

SOLOMON_TOKEN = 'none'
JUGGLER_TOKEN = 'none'


def get_production_version():
    r = requests.post(HAMSTER_VERSION_URL)
    tag = 'arcadia.yandex.ru/arc/tags/vins/stable-'
    version = r.text[r.text.find(tag)+len(tag):]
    return version[:version.find('/')]


def get_all_scenario_configs():
    from alice.megamind.library.config.scenario_protos.config_pb2 import TScenarioConfig
    configs = []
    version = get_production_version()
    svn_url = 'svn+ssh://arcadia.yandex.ru/arc/tags/vins/stable-' + version + '/arcadia/alice/megamind/configs/production/scenarios'
    svn.Arcadia.checkout(svn_url, CONFIGS_FOLDER_NAME)
    for config_file_name in os.listdir(CONFIGS_FOLDER_NAME):
        if 'pb.txt' in config_file_name:
            f = open(CONFIGS_FOLDER_NAME + '/' + config_file_name, 'r')
            config = TScenarioConfig()
            text_format.Parse(f.read(), config)
            configs.append(config)
    return configs


def get_common_headers(token):
    auth_header_value = 'OAuth ' + token
    return {
        'Content-Type': 'application/json',
        'Authorization': auth_header_value
    }


def filter_top_scenarios(all_scenarios):
    body = {
        'from': (datetime.utcnow() - timedelta(minutes=35)).strftime('%Y-%m-%dT%H:%M:%SZ'),
        'to': (datetime.utcnow() - timedelta(minutes=30)).strftime('%Y-%m-%dT%H:%M:%SZ'),
        'program': '{name=\"wins_per_second\", host=\"cluster\", service=\"server\", cluster=\"prod\"}'
    }
    r = requests.post(SOLOMON_DATA_URL, headers=get_common_headers(SOLOMON_TOKEN), json=body)
    scenarios_rps = []
    for item in r.json()['vector']:
        scenario_name = item['timeseries']['labels']['scenario_name']
        logging.info("Got scenario from wins_per_second: " + str(scenario_name))
        values = [v for v in item['timeseries']['values'] if v != u'NaN']
        if len(values) == 0 or scenario_name == 'any_scenario':
            continue
        logging.info(values)
        avg_rps = sum(values)/len(values)
        logging.info("Avg rps = " + str(avg_rps))
        scenarios_rps.append([avg_rps, scenario_name])
    scenarios_rps.sort(reverse=True)
    top_scenarios = [x[1] for x in scenarios_rps][:TOP_SCENARIOS_COUNT]
    target_configs = [c for c in all_scenarios if c.Name in top_scenarios]
    return target_configs


def get_tg_from_login(login):
    r = requests.post(STAFF_LOGIN_TO_TG_URL + login, headers=get_common_headers(SOLOMON_TOKEN))
    return '@' + r.json()['result'][0]['telegram_accounts'][0]['value']


def get_tg_from_logins(logins):
    result_logins = []
    for login in logins:
        result_logins.append(get_tg_from_login(login))
    return result_logins


def get_tg_from_slugs(slugs, service_name):
    result_logins = []
    r = requests.get(ABC_SLUG_URL + service_name, headers=get_common_headers(SOLOMON_TOKEN))
    for person in r.json():
        if person is None or person['person'] is None or person['schedule'] is None:
            continue
        if person['schedule']['slug'] in slugs:
            result_logins.append(get_tg_from_login(person['person']['login']))
    return result_logins


def make_message_for_alert(config):
    result_message = ''
    responsible_logins = get_tg_from_logins(config.Responsibles.Logins)
    result_message += 'Responsibles: ' + (' '.join(responsible_logins) if len(responsible_logins) > 0 else 'not found!') + '\n'
    for abc_service in config.Responsibles.AbcServices:
        slug_logins = get_tg_from_slugs(abc_service.DutySlugs, abc_service.Name)
        if len(abc_service.Name) > 0:
            result_message += ABC_SERVICE_URL + abc_service.Name + ' -> slug: ' + (' '.join(slug_logins) if len(slug_logins) > 0 else 'not found!') + '\n'
    return result_message


def create_graph_link(scenario_name):
    return 'https://solomon.yandex-team.ru/?project=megamind&cluster=prod&service=server&l.scenario_name=' + scenario_name.upper() +\
           '&l.name=scenario.protocol.responses_per_second&l.host=cluster&l.request_type=*&l.response_type=error*&graph=auto'


def create_juggler_check(config):
    body = {
        "project": "alice_operation",
        "host": "megamind",
        "service": SERVICE_PREFIX + config.Name,
        "refresh_time": 60,
        "ttl": 0,
        "description": "for generated alert",
        "flaps": {
            "stable_time": 120,
            "critical_time": 180,
            "boost_time": 0
        },
        "tags": [
            "monitoring_project_id_megamind",
            "monitoring_alert_id_" + SERVICE_PREFIX + config.Name
        ],
        "meta": {
            "urls": [
                {
                    "title": "Solomon alert link",
                    "url": "https://solomon.yandex-team.ru/admin/projects/megamind/alerts/" + SERVICE_PREFIX + config.Name,
                    "type": "monitoring_alert"
                }
            ]
        }
    }
    logging.info('Creating juggler check: ' + SERVICE_PREFIX + config.Name)
    requests.post(JUGGLER_CREATE_OR_UPDATE_CHECK_URL, headers=get_common_headers(JUGGLER_TOKEN), json=body)


def create_alert(config):
    message = make_message_for_alert(config)
    program = 'let errors = {project=\"megamind\", cluster=\"prod\", service=\"server\", scenario_name=\"' + config.Name.upper() +\
              '\", name=\"scenario.protocol.responses_per_second\", host=\"cluster\", request_type=\"*\", response_type=\"error*\"};' +\
              'let total = {project=\"megamind\", cluster=\"prod\", service=\"server\", scenario_name=\"' + config.Name.upper() +\
              '\", name=\"scenario.protocol.responses_per_second\", host=\"cluster\", request_type=\"*\"};' +\
              'let avg_total = avg(group_lines(\"sum\", total));' +\
              'let avg_errors = avg(group_lines(\"sum\", errors));' +\
              'let percent = avg_errors / avg_total;' +\
              'let threshold = avg_total > 5000 ? 0.001 : (avg_total > 1000 ? 0.003 : (avg_total > 100 ? 0.03 : (avg_total > 30 ? 0.05 : 0.3)));' +\
              'alarm_if(percent >= threshold);' +\
              'alarm_if(avg_errors >= 1.5);'
    body = {
        'id': SERVICE_PREFIX + config.Name,
        'projectId': 'megamind',
        'name': '[Generated alert] Scenario ' + config.Name + ' errors',
        'channels': [
            {
                'config': {
                    'notifyAboutStatuses': ['OK', 'ALARM'],
                    'repeatDelaySecs': 0
                },
                'id': 'Juggler-2'
            }
        ],
        'type': {
            'expression': {
                'program': program,
                'checkExpression': ''
            }
        },
        'annotations': {
            'description': message,
            'graphLink': create_graph_link(config.Name),
            'service': SERVICE_PREFIX + config.Name
        },
        'windowSecs': 30,
        'delaySecs': 0,
        'description': ''
    }

    logging.info('Creating alert: ' + SERVICE_PREFIX + config.Name)
    requests.post(SOLOMON_ALERTS_URL, headers=get_common_headers(SOLOMON_TOKEN), json=body)
    create_juggler_check(config)


def delete_current_alerts():
    r = requests.get(SOLOMON_ALERTS_URL + '?pageSize=100000', headers=get_common_headers(SOLOMON_TOKEN))
    for item in r.json()['items']:
        if item['id'].startswith(SERVICE_PREFIX):
            logging.info('Deleting alert: ' + item['id'])
            r = requests.delete(SOLOMON_ALERTS_URL + item['id'], headers=get_common_headers(SOLOMON_TOKEN))
    # check no generated alerts
    r = requests.get(SOLOMON_ALERTS_URL + '?pageSize=100000', headers=get_common_headers(SOLOMON_TOKEN))
    for item in r.json()['items']:
        if item['id'].startswith(SERVICE_PREFIX):
            raise common.errors.TaskError('Cannot delete all alerts. Check your solomon token')


class UpdateScenarioAlertsTask(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(sdk2.Parameters):
        tokens = sdk2.parameters.YavSecret(
            'YAV secret identifier (with optional version)',
            default='sec-01g009h4nshys0bjyrr9bbfvzt'
        )
        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        global SOLOMON_TOKEN
        global JUGGLER_TOKEN
        SOLOMON_TOKEN = self.Parameters.tokens.data()['solomon']
        JUGGLER_TOKEN = self.Parameters.tokens.data()['juggler']
        delete_current_alerts()
        for c in filter_top_scenarios(get_all_scenario_configs()):
            create_alert(c)
