import logging
import datetime
from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary

# SANDBOX
DIRECTORY = "monitoring"
METRICS_BINARY_FILENAME = 'metrics'

# YT
YT_DIRECTORY = "//home/metrics/metrics_executable/"
METRICS_EXECUTABLE_PREFIX = "metrics"
METRICS_EXECUTABLE_LINK_FILENAME = "metrics"
METRICS_TEST_POSTFIX = "_test"
TTL_IN_DAYS = 30

log = logging.getLogger(__name__)


class MetricsBinaryPackage(sdk2.Resource):
    pass


class MetricsBinaryRelease(sdk2.Task):
    """Binary release task https://a.yandex-team.ru/arc/trunk/arcadia/search/metrics/monitoring"""

    class Parameters(sdk2.Parameters):
        use_last_binary = sdk2.parameters.Bool("Use last binary", default=True)
        production = sdk2.parameters.Bool("Release Production binary (not test)", default=True)
        yt_proxies = sdk2.parameters.List("Yt proxies", default=["hahn", "arnold"])
        package = sdk2.parameters.Resource("Metrics Binary Package", resource_type=MetricsBinaryPackage)
        yt_path = sdk2.parameters.String("Path to executable", required=False)
        link_yt_path = sdk2.parameters.String("Link to setup", required=False)

    def fill_paths(self):
        executable_filename = METRICS_EXECUTABLE_PREFIX + "_{:%Y-%m-%d-%H-%M}".format(datetime.datetime.now())
        link_filename = METRICS_EXECUTABLE_LINK_FILENAME
        if not self.Parameters.production:
            executable_filename += METRICS_TEST_POSTFIX
            link_filename += METRICS_TEST_POSTFIX
        self.Parameters.yt_path = YT_DIRECTORY + executable_filename
        self.Parameters.link_yt_path = YT_DIRECTORY + link_filename

    def on_save(self):
        self.fill_paths()
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={'target': 'sandbox/metrics/bin', 'release': 'stable'}
            ).first().id
        log.info("on_save(). Yt path: {}. Link yt path: {}".format(self.Parameters.yt_path, self.Parameters.link_yt_path))

    def on_execute(self):
        path = sdk2.ResourceData(self.Parameters.package).path
        log.info('Synced ' + str(list(path.iterdir())))
        binary = str(path / DIRECTORY / METRICS_BINARY_FILENAME)
        link_yt_path = str(self.Parameters.link_yt_path)
        yt_path = str(self.Parameters.yt_path)
        log.info("on_execute(). Yt path: {}. Link yt path: {}".format(yt_path, link_yt_path))
        for proxy in self.Parameters.yt_proxies:
            deprecate_by_link(proxy, link_yt_path, sdk2.Vault.data('metrics_yt_token'))
            upload_to_yt(binary, proxy, yt_path, sdk2.Vault.data('metrics_yt_token'))
            set_link(yt_path, link_yt_path, proxy, sdk2.Vault.data('metrics_yt_token'))


def deprecate_by_link(yt_proxy, link_yt_path, yt_token, ttl_days=TTL_IN_DAYS):
    import yt.wrapper as yt
    # Only system modules and these from Sandbox's virtual environment are allowed to be imported at top level
    yt.config.set_proxy(yt_proxy)
    yt.config["token"] = yt_token
    if yt.exists(link_yt_path) and yt.get(link_yt_path + "&/@type") == "link":
        target_yt_path = yt.get(link_yt_path + "&/@target_path")
        log.info("setting ttl for {}".format(target_yt_path))

        expiration_time = (datetime.datetime.now() + datetime.timedelta(days=ttl_days)).isoformat()
        yt.set(target_yt_path + "/@expiration_time", expiration_time)


def upload_to_yt(binary, yt_proxy, yt_path, yt_token):
    import yt.wrapper as yt
    yt.config.set_proxy(yt_proxy)
    yt.config["token"] = yt_token
    yt.smart_upload_file(binary,
                         yt_path,
                         placement_strategy="replace")
    log.info("uploaded to {} at {}".format(yt_proxy, yt_path))


def set_link(target_path, link_path, yt_proxy, yt_token):
    import yt.wrapper as yt
    yt.config.set_proxy(yt_proxy)
    yt.config["token"] = yt_token
    if not yt.exists(link_path) or yt.get(link_path + "&/@type") == "link":
        log.info("linking {} to {}".format(link_path, target_path))
        yt.link(target_path, link_path, force=True)
