import logging
import json
from sandbox import sdk2
import requests
from sandbox.sdk2.service_resources import SandboxTasksBinary


class MetricsParserPlatformPackage(sdk2.Resource):
    pass


class MetricsParserPlatformArcadiaRelease(sdk2.Task):
    """Binary release parsers task https://a.yandex-team.ru/arc/trunk/arcadia/search/scraper/parser_platform"""

    class Parameters(sdk2.Parameters):
        use_last_binary = sdk2.parameters.Bool("Use last binary", default=True)
        default_yt_path = '//home/qe/scraper/testing/ppp/ppp'
        yt_path = sdk2.parameters.String("Yt path", default=default_yt_path)
        yt_proxies = sdk2.parameters.List("Yt proxies", default=["hahn", "arnold"])
        package = sdk2.parameters.Resource("PPP Package", resource_type=MetricsParserPlatformPackage)

    def on_save(self):
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={'target': 'sandbox/metrics/bin', 'release': 'stable'}
            ).first().id

    def on_execute(self):

        # Do not forget to update sandbox binary when updating on_execute and called methods:
        # ya make
        # bin/bin upload --attr target=sandbox/metrics/bin --attr release=stable

        path = sdk2.ResourceData(self.Parameters.package).path
        logging.info('Synced ' + str(list(path.iterdir())))
        binary = path / 'ppp'
        for proxy in self.Parameters.yt_proxies:
            upload_to_yt(str(binary), proxy, self.Parameters.yt_path, sdk2.Vault.data('metrics_yt_token'))

        upload_configs(path / 'search_engine_config',
                       sdk2.Vault.data('scraper_robot_token'))


def upload_to_yt(binary, yt_proxy, yt_path, yt_token):
    import yt.wrapper as yt

    yt.config.set_proxy(yt_proxy)
    yt.config["token"] = yt_token
    yt.smart_upload_file(binary,
                         yt_path,
                         placement_strategy="replace")
    logging.info("uploaded to {} at {}", yt_proxy, yt_path)


def upload_configs(search_engine_config_path, scraper_token):

    configs = []
    headers = {
        "Authorization": "OAuth {}".format(scraper_token),
        "Accept": "application/json",
        "Content-type": "application/json"
    }

    for config_path in search_engine_config_path.iterdir():
        name = config_path.name
        if not name.endswith(".json"):
            logging.info("skip " + config_path)
            continue

        id, format = name.split(".")
        logging.info("id: " + id)

        content = None
        with config_path.open('r') as f:
            content = json.load(f)
        content['id'] = id
        configs.append(content)
    upload_configs_to_scraper(configs, headers)
    upload_configs_to_parsing_service(configs, headers)


def upload_configs_to_scraper(configs, headers):
    scraper_host_prod = "https://scraper.yandex-team.ru"
    scraper_host_testing = "https://scraper-dev.metrics.yandex-team.ru"
    path = "/api/scraper/engines/config/update"

    for host in [scraper_host_prod, scraper_host_testing]:
        full_path = host + path
        for config in configs:
            r = requests.post(
                full_path,
                headers=headers,
                data=json.dumps(config),
                verify=False
            )
            if r.status_code == 200:
                logging.info("Updated: {} on {}".format(config, host))
            else:
                raise Exception(
                    "response code {}: {} for query '{}'.".format(r.status_code, r.text, config)
                )


def upload_configs_to_parsing_service(configs, headers):
    parsing_service_host_prod = "http://serp-parser.metrics.yandex-team.ru"
    parsing_service_host_testing = "http://serp-parser-dev.metrics.yandex-team.ru"
    path = "/api/search-engine-configs/update-batch"

    for host in [parsing_service_host_prod, parsing_service_host_testing]:
        full_path = host + path

        r = requests.post(
            full_path,
            headers=headers,
            data=json.dumps(configs),
            verify=False
        )

        if r.status_code == 200:
            for config in configs:
                logging.info("Updated: {} on Parsing Service".format(config['id']))
        else:
            config_ids = ", ".join(config['id'] for config in configs)
            raise Exception(
                "response code {}: {} on Parsing Service for query: {}.".format(r.status_code, r.text, config_ids)
            )
