import os
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.juggler import jclient
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask


@with_parents
class BirthdayReminder(BaseMetrikaTask):
    class Parameters(CommonParameters):
        description = 'Send birthday remind email to staff group members and head'
        kill_timeout = 600

        with sdk2.parameters.Group("Secret"):
            staff_token = sdk2.parameters.YavSecret("Staff token yav secret", required=True, default='sec-01cq6h07rwpqmqzb15y08jbs5q')
            staff_token_key = sdk2.parameters.String("Key from secret", required=True, default='admin_oauth_token')

        custom_settings = sdk2.parameters.Bool("Set custom settings", default=False)

        with sdk2.parameters.RadioGroup("Check staff group list or login list", required=True) as group_or_login:
            group_or_login.values["groups"] = "Groups"
            group_or_login.values["logins"] = "Logins"

            with group_or_login.value["groups"]:
                group_list = sdk2.parameters.List("List of staff groups to check birthday (e.g. yandex_monetize_metrica)", required=True)

                with custom_settings.value[True]:
                    group_dst_list = sdk2.parameters.List("List of email who will receive message. If empty, message will be sent to group members and head.")

            with group_or_login.value["logins"]:
                login_list = sdk2.parameters.List("List of staff logins to check birthday", required=True)
                login_dst_list = sdk2.parameters.List("List of email who will receive message.", required=True)

        with custom_settings.value[True]:
            with sdk2.parameters.Group("Custom settings"):
                days_threshold = sdk2.parameters.Integer("Days threshold, during which birthday can be found", default=7)
                send_email = sdk2.parameters.Bool("Email sending or just debug output", default=True)

        with sdk2.parameters.Group("Monitoring"):
            juggler_host = sdk2.parameters.String('"host" parameter for juggler check', required=True, default="sandbox-birthday-reminder")
            juggler_service = sdk2.parameters.String('"service" parameter from juggler check', required=True)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        from metrika.admin.python.scripts.birthday_reminder import lib

        logging.debug("Run reminder")
        try:
            os.environ["ROBOT_METRIKA_ADMIN_OAUTH"] = self.Parameters.staff_token.data().get(self.Parameters.staff_token_key)
            lib.main(
                group_list=self.Parameters.group_list,
                login_list=self.Parameters.login_list,
                days_threshold=self.Parameters.days_threshold,
                dry_run=not self.Parameters.send_email,
                dst=self.Parameters.group_dst_list or self.Parameters.login_dst_list,
            )
        except Exception as e:
            url = "https://sandbox.yandex-team.ru/task/%s" % self.id
            message = "Exception in task %s, error: %s" % (url, str(e))
            jclient.send_events_to_juggler(
                host=self.Parameters.juggler_host,
                service=self.Parameters.juggler_service,
                status="CRIT",
                description=message,
            )
            raise
        else:
            jclient.send_events_to_juggler(
                host=self.Parameters.juggler_host,
                service=self.Parameters.juggler_service,
                status="OK",
                description="OK",
            )
