import logging

import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as sb_binary_task
import sandbox.projects.common.juggler as sb_juggler

import sandbox.projects.metrika.utils.base_metrika_task as bmt
import sandbox.projects.metrika.utils.settings as settings
import sandbox.projects.metrika.utils as utils

BISHOP_ENVIRONMENT_CHOICES = [
    ('production', 'production'),
    ('testing', 'testing'),
]


@bmt.with_parents
class MetrikaBishopBrokenConfigurationsChecker(bmt.BaseMetrikaTask):
    class Parameters(utils.CommonParameters):
        environment = sdk2.parameters.String(
            'Environment',
            required=True,
            choices=BISHOP_ENVIRONMENT_CHOICES,
        )
        token = sdk2.parameters.YavSecret(
            'Token',
            required=True,
            default='{}#bishop_token'.format(settings.rmb_yav_uuid),
        )
        api_url = sdk2.parameters.String(
            'Bishop api url',
            required=False,
        )
        send_event_to_juggler = sdk2.parameters.Bool(
            'Send event to juggler',
            default=True,
        )
        _binary = sb_binary_task.binary_release_parameters_list(stable=True)

    @property
    def tags(self):
        return [
            'BISHOP_BROKEN_CONFIGURATIONS_CHECKER',
            self.Parameters.environment,
        ]

    def on_execute(self):
        import metrika.pylib.bishop as bp

        separator = '=' * 40

        logging.info('%s ON_EXECUTE STARTED %s', separator, separator)

        self.Parameters.tags = self.tags

        secret = self.Parameters.token.data()
        token = secret[self.Parameters.token.default_key]

        client = bp.Bishop(
            token=token,
            url=self.Parameters.api_url or None,
            environment=self.Parameters.environment,
        )

        if self.Parameters.environment == 'production':
            bishop_url = 'https://bishop.mtrs.yandex-team.ru'
        else:
            bishop_url = 'https://bishop-test.mtrs.yandex-team.ru'

        status = 'OK'
        message = 'There is no broken configurations'
        message_tail = 'Please visit {}/system?part=broken_configs'.format(bishop_url)

        try:
            broken_configurations = client.get_broken_configurations()
        except Exception as err:
            message = 'failed to get broken configurations from bishop: {}'.format(err)
            logging.exception(message)
            status = 'CRIT'
        else:
            if broken_configurations:
                status = 'CRIT'
                message = ', '.join(sorted(set([bc.program for bc in broken_configurations])))
                message = 'There is broken configurations for next programs: {}. {}'.format(
                    message,
                    message_tail,
                )

        juggler_event = {
            'host': 'metrika.bishop.schedulers.{}'.format(self.Parameters.environment),
            'service': 'broken_configurations',
            'status': status,
            'description': message,
        }

        if self.Parameters.send_event_to_juggler:
            sb_juggler.jclient.send_events_to_juggler(**juggler_event)

        logging.info('%s ON_EXECUTE FINISHED %s', separator, separator)
