import logging

import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as sb_binary_task
import sandbox.projects.common.juggler as sb_juggler

import sandbox.projects.metrika.utils.base_metrika_task as bmt
import sandbox.projects.metrika.utils.settings as settings
import sandbox.projects.metrika.utils as utils

BISHOP_ENVIRONMENT_CHOICES = [
    ('production', 'production'),
    ('testing', 'testing'),
]

EXTERNAL_DATA_TYPE_CHOICES = [
    ('vault', 'vault'),
    ('hosts', 'hosts'),
    ('networks', 'networks'),
    ('yc_clusters', 'yc_clusters'),
]


@bmt.with_parents
class MetrikaBishopExternalDataRefresher(bmt.BaseMetrikaTask):
    class Parameters(utils.CommonParameters):
        external_data_type = sdk2.parameters.String(
            'External data type',
            required=True,
            choices=EXTERNAL_DATA_TYPE_CHOICES,
        )
        environment = sdk2.parameters.String(
            'Environment',
            required=True,
            choices=BISHOP_ENVIRONMENT_CHOICES,
        )
        token = sdk2.parameters.YavSecret(
            'Token',
            required=True,
            default='{}#bishop_token'.format(settings.rmb_yav_uuid),
        )
        api_url = sdk2.parameters.String(
            'Bishop api url',
            required=False,
        )
        send_event_to_juggler = sdk2.parameters.Bool(
            'Send event to juggler',
            default=True,
        )
        connect_timeout = sdk2.parameters.Integer(
            'Connect timeout',
            required=True,
            default=5,
        )
        operation_timeout = sdk2.parameters.Integer(
            'Operation timeout',
            required=True,
            default=180,
        )
        _binary = sb_binary_task.binary_release_parameters_list(stable=True)

    @property
    def tags(self):
        return [
            'BISHOP_EXTERNAL_DATA_REFRESHER',
            self.Parameters.external_data_type,
            self.Parameters.environment,
        ]

    def on_execute(self):
        import metrika.pylib.bishop as bp

        separator = '=' * 40

        logging.info('%s ON_EXECUTE STARTED %s', separator, separator)

        self.Parameters.tags = self.tags

        secret = self.Parameters.token.data()
        token = secret[self.Parameters.token.default_key]

        client = bp.Bishop(
            token=token,
            url=self.Parameters.api_url or None,
            environment=self.Parameters.environment,
        )

        status = 'OK'
        base_error_message = 'failed to update {}'.format(self.Parameters.external_data_type)
        base_success_message = '{} updated'.format(self.Parameters.external_data_type)

        try:
            response = client.refresh(
                self.Parameters.external_data_type,
                timeout=(
                    self.Parameters.connect_timeout,
                    self.Parameters.operation_timeout,
                )
            )
        except Exception as err:
            message = '{}: {}'.format(base_error_message, err)
            logging.exception(message)
            status = 'CRIT'
        else:
            if response.result:
                message = '{}: {}'.format(
                    base_success_message,
                    ', '.join(response.messages),
                )
                logging.info(message)
            else:
                message = '{}: response result is not True: {}'.format(
                    base_error_message,
                    ', '.join(response.errors),
                )
                logging.error(message)
                status = 'CRIT'

        juggler_event = {
            'host': 'metrika.bishop.schedulers.{}'.format(self.Parameters.environment),
            'service': 'refresh_{}'.format(self.Parameters.external_data_type),
            'status': status,
            'description': message,
        }

        if self.Parameters.send_event_to_juggler:
            sb_juggler.jclient.send_events_to_juggler(**juggler_event)

        logging.info('%s ON_EXECUTE FINISHED %s', separator, separator)
