import logging
from itertools import chain

import sandbox.projects.common.binary_task as sb_binary_task
import sandbox.projects.common.juggler as sb_juggler
import sandbox.sdk2 as sdk2

import sandbox.projects.metrika.utils as utils
import sandbox.projects.metrika.utils.base_metrika_task as bmt
import sandbox.projects.metrika.utils.settings as settings

BISHOP_ENVIRONMENT_CHOICES = [
    ('production', 'production'),
    ('testing', 'testing'),
]


@bmt.with_parents
class MetrikaBishopLinkARs(bmt.BaseMetrikaTask):
    name = 'METRIKA_BISHOP_LINK_ARS'

    class Parameters(utils.CommonParameters):
        environment = sdk2.parameters.String(
            'Environment',
            required=True,
            choices=BISHOP_ENVIRONMENT_CHOICES,
        )
        token = sdk2.parameters.YavSecret(
            'Token',
            required=True,
            default='{}#bishop_token'.format(settings.rmb_yav_uuid),
        )
        api_url = sdk2.parameters.String(
            'Bishop api url',
            required=False,
        )
        send_event_to_juggler = sdk2.parameters.Bool(
            'Send event to juggler',
            default=True,
        )

        _binary = sb_binary_task.binary_release_parameters_list(stable=True)

    @property
    def tags(self):
        return [
            'BISHOP_LINK_ARS',
            self.Parameters.environment,
        ]

    def on_execute(self):
        import metrika.pylib.bishop as bp

        separator = '=' * 40

        logging.info('%s ON_EXECUTE STARTED %s', separator, separator)

        self.Parameters.tags = self.tags

        secret = self.Parameters.token.data()
        token = secret[self.Parameters.token.default_key]

        client = bp.Bishop(
            token=token,
            url=self.Parameters.api_url or None,
            environment=self.Parameters.environment,
        )

        message = status = 'OK'
        try:
            ars = client.link_ars().data
            logging.debug(ars)
        except Exception as err:
            message = 'Failed to link ARs in bishop: {}'.format(err)
            status = 'CRIT'
        else:
            if ars['not_linked']:
                not_linked = ', '.join(sorted(set(chain.from_iterable(ars['not_linked'].values()))))
                message = 'Failed to link ARs to tickets, check task\'s logs and access for robot-metrika-bishop: {}'.format(not_linked)
                status = 'CRIT'

        juggler_event = {
            'host': 'metrika.bishop.schedulers.{}'.format(self.Parameters.environment),
            'service': 'link_ars',
            'status': status,
            'description': message,
        }

        if self.Parameters.send_event_to_juggler:
            sb_juggler.jclient.send_events_to_juggler(**juggler_event)

        logging.info('%s ON_EXECUTE FINISHED %s', separator, separator)
