# coding=utf-8
import gzip
import logging
import os
import tempfile

import requests

from sandbox import common, sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import base_metrika_task, resource_types
from sandbox.projects.metrika.utils.mixins import dyn_resource, juggler_reporter


class MetrikaClickhouseIpMapperDictionary(sdk2.Resource):
    any_arch = True
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE
    releasable = True
    releasers = resource_types.METRIKA_RELEASERS
    share = True
    last_modified = sdk2.Attributes.String("Last-Modified HTTP Header value", default="")


@base_metrika_task.with_parents
class MetrikaBuildClickhouseIpMapperDictionary(base_metrika_task.BaseMetrikaTask, juggler_reporter.JugglerReporterMixin, dyn_resource.DynResourceMixin):
    juggler_host = 'metrika-sandbox'

    class Parameters(utils.CommonParameters):
        url = sdk2.parameters.String(
            "ip mapper dict source",
            default="http://export.geogen.yandex-team.ru/IPv6/ip_origin.gz",
            required=True,
        )

        resource_filename = sdk2.parameters.String(
            "Result dictionary filename",
            default="ip_mapper.txt",
            required=True,
        )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        logging.info("Started")

        self.create_resource()

        logging.info("Finished")

    def create_resource(self):
        with self.memoize_stage.create_resource:
            try:
                logging.info("Requesting %s" % self.Parameters.url)
                response = requests.get(self.Parameters.url, stream=True)
                logging.info("Response headers: %s" % response.headers)

                resource_data = self.get_resource_data(response.headers['Last-Modified'])
                self.prepare_remote_content(response)
                logging.info("Making resource data ready")
                resource_data.ready()
            except Exception as error:
                message = "Unhandled exception: %s" % error
                raise common.errors.TaskFailure(message)

    def get_resource_data(self, last_modified):
        resource = MetrikaClickhouseIpMapperDictionary(
            self,
            "Metrika ClickHouse IP Mapper Dictionary text file",
            self.Parameters.resource_filename,
            last_modified=last_modified,
        )
        resource_data = sdk2.ResourceData(resource)
        return resource_data

    def prepare_remote_content(self, response):
        import netaddr
        logging.info("Downloading remote content")
        with tempfile.NamedTemporaryFile(mode='w+b') as temp:
            for chunk in response.iter_content(chunk_size=4096):
                if chunk:
                    temp.write(chunk)

            temp.flush()
            temp.seek(0)
            logging.info("Content downloaded. Size: %s" % os.stat(temp.name).st_size)

            logging.info("Gunzipping and parsing")
            gzf = gzip.GzipFile(mode='rb', fileobj=temp)
            out_template = '%s\t%s\tZZ\n'

            with open(self.Parameters.resource_filename, 'w+') as parsed:
                for line in gzf:
                    spl = line.strip().split()
                    net_as = spl[-1]
                    ips = spl[0].split('-')
                    ip1 = ips[0]
                    ip2 = ips[1]
                    ip_range = netaddr.IPRange(ip1, ip2)
                    for net in ip_range.cidrs():
                        parsed.write(out_template % (net, net_as))
        logging.info("Content prepared")
