# coding=utf-8

import logging
import random
import time
from urllib import quote_plus

import requests
from requests.packages.urllib3.util.retry import Retry

from sandbox.common.errors import TaskError
from sandbox.projects.metrika.admins.clickhouse.metrika_mdb_ch_upgrade import MetrikaMdbChUpgrade, MetrikaMdbChUpgradeParameters
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.auth import MdbAuth
from sandbox.projects.metrika.utils.base_metrika_task import BaseMetrikaTask
from sandbox.sdk2 import parameters


class ClickhouseMdbAcceptance(BaseMetrikaTask):
    """
    Базовая таска для приемки ClickHouse в MDB для кластеров
    """
    CLUSTERS = [
        ('mtaggr-test', 'statistics', 'hour2', 'StartDate'),
        ('publishers-test', 'publishers', 'publishers2_merge', 'HitEventDate'),
        ('expenses-test', 'expenses', 'expenses', 'Date'),
        ('webvisor-test', 'wv', 'wv_forms_layer_1', 'EventDate'),
        ('skadnetwork-test', 'skadnetwork', 'skad_postbacks_all', 'EventDate'),
    ]

    CHECK_QUERY = "SELECT count() as count FROM {db}.{table} WHERE {column} IN (today(), yesterday())"
    INTERVAL = 60
    MAX_ITERATIONS = 6 * 60  # 6 часов при INTERVAL = 60

    class Parameters(MetrikaMdbChUpgradeParameters):
        kill_timeout = 24 * 60 * 60  # Ждём результата не более суток

        with parameters.Group("Секреты пользователя для доступа в ClickHouse") as test_secrets_group:
            robot_test_tokens_secret = parameters.YavSecret("Секрет с паролями", required=True, default=settings.yav_uuid)
            robot_test_token_key = parameters.String("Ключ пароля в секрете", required=True, default="clickhouse-password")

        ch_cluster = parameters.String("Имя кластера", required=True)
        ch_database = parameters.String("Имя базы данных", required=True)
        ch_table = parameters.String("Имя таблицы", required=True)
        ch_column = parameters.String("Имя столбца для фильтрации по дате", required=True)
        ch_user = parameters.String("Имя пользователя", required=True, default="robot-metrika-test")
        pre_check = parameters.Bool("Запустить преварительную проверку работоспособности", required=True, default=True)
        upgrade = parameters.Bool("Апгрейдить/даунгрейдить версию ClickHouse", required=True, default=True)

    def on_execute(self):
        from metrika.pylib.yc.clickhouse import ManagedClickhouse
        self._ch_password = self.Parameters.robot_test_tokens_secret.data().get(self.Parameters.robot_test_token_key)

        token = self.Parameters.robot_admin_tokens_secret.data().get(self.Parameters.robot_admin_token_key)
        self.cluster = ManagedClickhouse(token=token).cluster_by_name(self.Parameters.ch_cluster, self.Parameters.mdb_folder)
        self.servers = []

        self._get_servers()

        if self.Parameters.pre_check:
            self.run_check(pre_check=True)
        if self.Parameters.upgrade:
            self.run_upgrade_task()
        self.run_check()

    def run_check(self, pre_check=False):
        query = self.CHECK_QUERY.format(db=self.Parameters.ch_database, table=self.Parameters.ch_table, column=self.Parameters.ch_column)
        previous_check_result = self._execute_query(query)
        iteration = 0
        while True:
            iteration += 1
            check_result = self._execute_query(query)
            success = self.compare_results(previous_check_result, check_result)
            if success:
                return
            if iteration > self.MAX_ITERATIONS:
                if pre_check:
                    error_msg = "Не удалось получить положительный результат предварительной проверки. Похоже и так ничего не работает."
                else:
                    error_msg = "Не удалось получить положительный результат проверки."
                raise TaskError(error_msg)
            previous_check_result = check_result
            time.sleep(self.INTERVAL)

    def compare_results(self, previous, current):
        return previous[0]['count'] < current[0]['count']

    def run_upgrade_task(self):
        self.run_subtasks([(
            MetrikaMdbChUpgrade,
            dict(
                mdb_folder=self.Parameters.mdb_folder,
                ch_cluster=self.Parameters.ch_cluster,
                ch_version=self.Parameters.ch_version,
                robot_admin_tokens_secret=self.Parameters.admin_secrets_group.robot_admin_tokens_secret,
                robot_admin_token_key=self.Parameters.admin_secrets_group.robot_admin_token_key
            )
        )])

    def _get_servers(self):
        try:
            self.servers = [x['name'] for x in self.cluster.list_hosts() if x['type'] == 'CLICKHOUSE']
        except Exception:
            error_msg = "Ошибка в процессе получения хостов кластера."
            logging.exception(error_msg)
            raise TaskError(error_msg)
        if not self.servers:
            error_msg = "Не найдено хостов ClickHouse. Проверьте состояние кластера в MDB."
            raise TaskError(error_msg)

    @property
    def _ch_session(self):
        try:
            return self.__ch_session
        except AttributeError:
            self.__ch_session = requests.session()
            self.__ch_session.auth = MdbAuth(self.Parameters.ch_user, self._ch_password)
            adapter = requests.adapters.HTTPAdapter(max_retries=Retry(total=2, method_whitelist=frozenset(['GET']), status_forcelist=(500, 502, 504), backoff_factor=1))
            self.__ch_session.mount("http", adapter)
            self.__ch_session.mount("https", adapter)
            self.__ch_session.verify = False
        return self.__ch_session

    def _execute_query(self, query):
        logging.info("Выполняем запрос: {}".format(query))
        host = random.choice(self.servers)
        logging.info("Выбранный хост: {}".format(host))
        try:
            protocol, port = ('https', 8443) if host.endswith('db.yandex.net') else ('http', 8123)
            resp = self._ch_session.get(
                "{protocol}://{host}:{port}/?query=".format(protocol=protocol, host=host, port=port) +
                quote_plus("{query} FORMAT JSON".format(query=query))
            )
            logging.info("Ответ: {}".format(resp.text))
            resp.raise_for_status()
            result = resp.json().get("data")
            return result
        except Exception:
            logging.exception("Ошибка при выполнении запроса.")
            return None
