# coding=utf-8
import logging
import re

import os
import six

from sandbox import sdk2
from sandbox.common import utils as common_utils
from sandbox.common.patterns import singleton_property
from sandbox.common.types import task as task_type
from sandbox.common.urls import get_resource_link
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.admins.clickhouse import clickhouse_b2b_api_tests_pipeline, clickhouse_b2b_test_run, clickhouse_mdb_acceptance, clickhouse_regression_tests,\
    clickhouse_statbox_traf_tests_pipeline, clickhouse_statbox_traf_tests_run, clickhouse_test_queues_processing
from sandbox.projects.metrika.admins.clickhouse.clickhouse_b2b_test_requests_download import handles
from sandbox.projects.metrika.core import appmetrica_core_b2b_tests_run, metrika_core_arcadia_tests_run, metrika_core_b2b_tests_run
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import conductor, parameters as metrika_parameters, resource_types, vcs
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.mixins import subtasks, console, releasable
from sandbox.projects.metrika.utils.pipeline import pipeline

REPORT_TTL = 730

package = "clickhouse-server"
metrika_cluster = "mtgiga"
appmetrica_cluster = "mtmobgiga"


@with_parents
class ClickHouseReleaseAcceptance(pipeline.PipelineBaseTask, releasable.ReleasableTaskMixin, console.BaseConsoleMixin):
    """
    Приёмка релиза ClickHouse
    """
    name = "CLICKHOUSE_RELEASE_ACCEPTANCE"

    @property
    def release_template(self):
        return sdk2.ReleaseTemplate(None, None, None, [task_type.ReleaseStatus.STABLE, task_type.ReleaseStatus.PRESTABLE, task_type.ReleaseStatus.TESTING])

    class Context(sdk2.Context):
        tests = []
        mdb_tasks = []
        queues_tasks = []

    class Parameters(utils.CommonParameters):
        description = "Приёмка релиза ClickHouse"

        test_version = sdk2.parameters.String("Тестируемая версия ClickHouse", required=True, default="", description="Задается релиз-инженером")

        ticket = metrika_parameters.TrackerIssue('Тикет предыдущей незавершенной приемки', required=False)

        stable_version_metrika = metrika_parameters.PackageVersion(
            "Текущая версия ClickHouse, которая считается стабильной для Метрики",
            package=package, host_group=metrika_cluster,
            required=True, description="Если не задана - получается по версии пакета {} с кластера {}".format(package, metrika_cluster)
        )
        stable_version_appmetrica = metrika_parameters.PackageVersion(
            "Текущая версия ClickHouse, которая считается стабильной для АппМетрики",
            package=package, host_group=appmetrica_cluster,
            required=True, description="Если не задана - получается по версии пакета {} с кластера {}".format(package, appmetrica_cluster)
        )

        config_stable_version = metrika_parameters.PackageVersion(
            "Стабильная версия clickhouse-server-metrika",
            package="clickhouse-server-metrika", host_group=metrika_cluster,
            required=True, description="Если не задана - получается по версии пакета {} с кластера {}".format(package, metrika_cluster)
        )
        config_test_version = metrika_parameters.PackageVersion(
            "Тестовая версия clickhouse-server-metrika",
            package="clickhouse-server-metrika", host_group=metrika_cluster,
            required=True, description="Если не задана - получается по версии пакета {} с кластера {}".format(package, metrika_cluster)
        )

        data_center_params = metrika_parameters.DataCenterParameters()

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_enqueue(self):
        self.Parameters.tags = [self.Parameters.test_version]

    def create_stages(self):
        return [
            (self.stage_create_issue, "Создание тикета"),
            (self.stage_prepare_tests, "Подготовка тестов"),
            (self.stage_run_tests, "Тестирование"),
        ]

    def stage_create_issue(self):
        summary = "Приёмка ClickHouse {}".format(self.Parameters.test_version)
        description = (
            "((https://wiki.yandex-team.ru/jandexmetrika/operations/Reglament-prijomki-reliza-ClickHouse/ Регламент приёмки релиза ClickHouse))\n"
            "(({} Задача))".format(self.link)
        )
        tags = [self.Parameters.test_version]

        if not self.Parameters.ticket:
            search_parameters = {
                "queue": "MTRSADMIN",
                "components": 45122,
                "type": "task",
                "tags": tags
            }
            create_parameters = search_parameters.copy()
            create_parameters.update({
                "summary": summary,
                "assignee": self.real_author,
                "description": description,
            })

            self.Context.issue_key = metrika_java_helper.MetrikaJavaHelper.find_or_create_issue(self.st_client, search_parameters, create_parameters, self.id).key
        else:
            ticket = self.st_client.issues[self.Parameters.ticket]
            ticket.update(summary=summary, description=description, tags=sorted(set(ticket.tags + tags)))
            self.Context.issue_key = ticket.key

        logging.info("Issue {} created.".format(self.Context.issue_key))

        self.set_info("Тикет: <a href=\"https://st.yandex-team.ru/{0}\">{0}</a>".format(self.Context.issue_key), do_escape=False)

    def stage_prepare_tests(self):
        self.Context.ch_binary_resource = self.find_clickhouse_binary(self.Parameters.test_version)
        self.Context.ch_stable_metrika_binary_resource = self.find_clickhouse_binary(self.Parameters.stable_version_metrika)
        self.Context.ch_stable_appmetrica_binary_resource = self.find_clickhouse_binary(self.Parameters.stable_version_appmetrica)
        self.comment('<a href="{}">Бинарь КХ</a>'.format(get_resource_link(self.Context.ch_binary_resource)))
        self.__update_issue_description()

    def find_clickhouse_binary(self, clickhouse_version):
        ch_binary_resource = resource_types.MetrikaClickhouseBinary.find(attrs=dict(version=clickhouse_version), state='READY').first()
        if ch_binary_resource is None:
            raise pipeline.PipelineError("Ресурс с версией {} бинаря ClickHouse не обнаружен. Выполните импорт.".format(self.Parameters.test_version))
        else:
            return ch_binary_resource.id

    def stage_run_tests(self):
        self.run_subtasks(self.tests, subtasks_variable=self.Context.tests, after_subtasks_enqueued=self.__update_issue_description, after_subtasks_waited=self.__update_issue_description)

    @staticmethod
    def __task_is_not_finished(task):
        return task.status not in utils.TASK_FINISHED_STATUSES

    def __update_issue_description(self):
        tests_table_start = "\n---\n#|\n"
        tests_table_end = "|#"
        tests_table = tests_table_start + self.__get_tests_table() + tests_table_end

        description = self.st_client.issues[self.Context.issue_key].description or ''
        if tests_table_start not in description:
            description += tests_table
        else:
            description = re.sub(utils.decode("(?s){}.*{}".format(tests_table_start.replace("|", "\\|"), tests_table_end.replace("|", "\\|"))), tests_table, description)

        self.st_client.issues[self.Context.issue_key].update(description=description)

    def __get_tests_table(self):
        return "".join(
            ClickHouseReleaseAcceptance.__get_tests_table_row(self, test[1].get("description"), test_task_id)
            for test, test_task_id in six.moves.zip_longest(self.tests, self.Context.tests)
        )

    def __get_tests_table_row(self, test_task_description, test_task_id):
        test_task = sdk2.Task[test_task_id] if test_task_id else None
        test_task_link = common_utils.get_task_link(test_task.id) if test_task else None
        return "|| {} | {} | {} | {} ||\n".format(
            self.__get_test(test_task_description, test_task_link),
            self.__get_statistics(test_task),
            self.__get_result(test_task),
            self.__get_result_icon(test_task))

    def __get_test(self, test_task_description, test_task_link):
        if not test_task_link:
            return test_task_description
        else:
            return "(({} {}))".format(test_task_link, test_task_description)

    def __get_statistics(self, test_task):
        if test_task:
            if test_task.type == clickhouse_regression_tests.ClickhouseRegressionTests:
                subtask = test_task.find(task_type=metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun).first()
                return self.__get_statistics_for_task(subtask)
            if test_task.type == clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline:
                faced_subtask = test_task.find(task_type=clickhouse_b2b_test_run.ClickHouseB2BTestRun, input_parameters={
                    clickhouse_b2b_test_run.ClickHouseB2BTestRun.Parameters.mobmetd_api_test.name: None
                }).first()
                mobmetd_subtask = test_task.find(task_type=clickhouse_b2b_test_run.ClickHouseB2BTestRun, input_parameters={
                    clickhouse_b2b_test_run.ClickHouseB2BTestRun.Parameters.faced_api_test.name: None
                }).first()
                return " ".join(self.__get_statistics_for_task(subtask, b2b_report=True) for subtask in [faced_subtask, mobmetd_subtask])
            if test_task.type == clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline:
                subtask = test_task.find(task_type=clickhouse_statbox_traf_tests_run.ClickHouseStatboxTrafTestRun).first()
                return self.__get_statistics_for_task(subtask)

        return self.__get_statistics_for_task(test_task)

    def __get_statistics_for_task(self, test_task, b2b_report=False):
        if not test_task or not getattr(test_task, "_get_view_model", None) or not test_task.Parameters.report_resource:
            return "-/-"
        else:
            return "(({} {}/{}))".format(
                test_task._get_view_model().allure_link if not b2b_report else self.__get_b2b_report_link(test_task),
                test_task.Parameters.test_results.get("success"),
                test_task.Parameters.test_results.get("total")
            )

    def __get_b2b_report_link(self, test_task):
        return "{}/site/response-report/index.html".format(sdk2.Resource[test_task.Parameters.report_resource.id].http_proxy)

    def __get_result(self, test_task):
        if not test_task or test_task.status not in utils.TASK_FINISHED_STATUSES:
            return "NOT FINISHED"
        if test_task.status in task_type.Status.Group.SUCCEED:
            return "SUCCESS"
        elif "OK" in test_task.Parameters.tags:
            return "MANUAL OK"
        else:
            return "FAILURE"

    def __get_result_icon(self, test_task):
        if not test_task or test_task.status not in utils.TASK_FINISHED_STATUSES:
            return "&#x2796;"
        if self.check_subtask(test_task):
            return "&#x2714;&#xFE0F;"
        else:
            return "&#x274C;"

    @staticmethod
    def __get_clickhouse_recipe_targets(path):
        recipe = "metrika/core/recipes/clickhouse"
        includes = [recipe]
        targets = []
        with vcs.mount_arc("arcadia-arc:/#trunk") as arcadia:
            for directory, _, files in os.walk(os.path.join(arcadia, path)):
                test_inc = "test.inc"
                if test_inc in files:
                    if recipe in open(os.path.join(directory, test_inc)).read():
                        includes.append(os.path.join(os.path.relpath(directory, arcadia), test_inc))
            for directory, _, files in os.walk(os.path.join(arcadia, path)):
                ya_make = "ya.make"
                if ya_make in files:
                    logging.debug("%s/%s", directory, ya_make)
                    ya_make_content = open(os.path.join(directory, ya_make)).read()
                    for include in includes:
                        if include in ya_make_content:
                            targets.append(os.path.relpath(directory, arcadia))
        return targets

    @singleton_property
    def tests(self):
        tests = [
            (
                clickhouse_regression_tests.ClickhouseRegressionTests,
                {
                    "description": "Регрессионные тесты релиза ClickHouse {}".format(self.Parameters.test_version),
                    clickhouse_regression_tests.ClickhouseRegressionTests.Parameters.use_existing_caas: False,
                    clickhouse_regression_tests.ClickhouseRegressionTests.Parameters.clickhouse_version: self.Parameters.test_version,
                    clickhouse_regression_tests.ClickhouseRegressionTests.Parameters.ch_config_version: self.Parameters.config_test_version,
                    clickhouse_regression_tests.ClickhouseRegressionTests.Parameters.data_center: self.Parameters.data_center
                }
            ),
            (
                clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline,
                {
                    "description": "Тесты API релиза ClickHouse {}".format(self.Parameters.test_version),
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.caas_parent: "days-32_sample-4",
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.faced_handles_include: (
                        '|'.join('^{}$'.format(handle['handle']) for handle in handles.REQUESTS_CONF['faced']['handles'] if handle['release_acceptance'])
                    ),
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.mobmetd_handles_include: (
                        '|'.join('^{}$'.format(handle['handle']) for handle in handles.REQUESTS_CONF['mobmetd']['handles'] if handle['release_acceptance'])
                    ),
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.portion: 0.9,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.limit: 10000,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.threads: 64,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.clickhouse_version_test: self.Parameters.test_version,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.clickhouse_version_ref: self.Parameters.stable_version_metrika,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.config_version_test: self.Parameters.config_test_version,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.config_version_ref: self.Parameters.config_stable_version,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.report_startrek: False,
                    clickhouse_b2b_api_tests_pipeline.ClickHouseB2bApiTestsPipeline.Parameters.data_center: self.Parameters.data_center
                }
            ),
            (
                clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline,
                {
                    # TODO там трэш со словарями
                    "description": "Тесты Трафа релиза ClickHouse {}".format(self.Parameters.test_version),
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.caas_parent: "days-32_sample-4",
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.clickhouse_version_test: self.Parameters.test_version,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.clickhouse_version_ref: self.Parameters.stable_version_metrika,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.config_version_test: self.Parameters.config_test_version,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.config_version_ref: self.Parameters.config_stable_version,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.report_startrek: False,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.tolerance: True,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.max_abs_diff: 1e-4,
                    clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.Parameters.data_center: self.Parameters.data_center
                }
            ),
            (
                appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun,
                {
                    "description": "Тесты Движка АппМетрики релиза ClickHouse {}".format(self.Parameters.test_version),
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.arcadia_url: "arcadia-arc:/#trunk",
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.test_clickhouse_resource: self.Context.ch_binary_resource,
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.stable_clickhouse_resource: self.Context.ch_stable_appmetrica_binary_resource,
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.force_test_scenario: True,
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.force_stable_scenario: True,
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.report_startrek: False,
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.fail_task_on_test_failure: True,
                    appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun.Parameters.report_ttl: REPORT_TTL
                }
            ),
            (
                metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun,
                {
                    "description": "Тесты Движка Метрики релиза ClickHouse {}".format(self.Parameters.test_version),
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.arcadia_url: "arcadia-arc:/#trunk",
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.test_clickhouse_resource: self.Context.ch_binary_resource,
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.stable_clickhouse_resource: self.Context.ch_stable_metrika_binary_resource,
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.force_test_scenario: True,
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.force_stable_scenario: True,
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.report_startrek: False,
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.fail_task_on_test_failure: True,
                    metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.Parameters.report_ttl: REPORT_TTL
                }
            )
        ]

        for target in self.__get_clickhouse_recipe_targets("metrika/core/programs") + self.__get_clickhouse_recipe_targets("metrika/java/api"):
            tests.append((
                metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun,
                {
                    "description": "Тесты {} релиза ClickHouse {}".format(target, self.Parameters.test_version),
                    metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun.Parameters.checkout_arcadia_from_url: "arcadia-arc:/#trunk",
                    metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun.Parameters.targets: target,
                    metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun.Parameters.report_startrek: False,
                    metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun.Parameters.fail_task_on_test_failure: True,
                    metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun.Parameters.definition_flags: "-DUSE_SB_CH={}".format(self.Context.ch_binary_resource),
                    metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun.Parameters.allure_report_ttl: REPORT_TTL
                }
            ))

        return tests

    def on_release(self, parameters_):
        if parameters_['release_status'] in ['testing', 'stable']:
            if parameters_['release_status'] == 'stable':
                failed_subtasks = [
                    task_id
                    for task_id in self.Context.mdb_tasks + self.Context.queues_tasks
                    if not self.check_subtask(sdk2.Task[task_id])
                ]
                if failed_subtasks:
                    raise subtasks.SubTasksError(failed_subtasks)

                if 'PROD' in self.Parameters.tags:
                    ch_bin = resource_types.MetrikaClickhouseBinary[self.Context.ch_binary_resource]
                    ch_bin.in_production = True
                    # а так же зарелизим ту таску, которая его создала - сейчас это импорт дистрибутива
                    self.server.release(
                        task_id=ch_bin.task_id,
                        type=task_type.ReleaseStatus.STABLE,
                        subject="{} {}".format(self.Parameters.description, self.Parameters.test_version)
                    )
                    return

            packages = [
                {
                    package_name: self.Parameters.test_version
                    for package_name in ["clickhouse-client", "clickhouse-common-static", "clickhouse-server"]
                }
            ]

            if self.Parameters.config_stable_version != self.Parameters.config_test_version:
                packages.append({"clickhouse-server-metrika": self.Parameters.config_test_version})

            for package in packages:
                conductor_ticket = conductor.Conductor(self).create_conductor_ticket(
                    package, self.Context.issue_key, no_autoinstall=True, projects=["metrika"], branch=parameters_['release_status']
                )
                self.set_info("Кондукторный тикет {0} в {1}: <a href=\"https://c.yandex-team.ru/tickets/{2}\">{2}</a>".format(
                    ', '.join(sorted(package)), parameters_['release_status'], conductor_ticket
                ), do_escape=False)
        elif parameters_['release_status'] == 'prestable':
            self.run_subtasks([
                (
                    clickhouse_test_queues_processing.ClickhouseTestQueuesProcessing,
                    {
                        clickhouse_test_queues_processing.ClickhouseTestQueuesProcessing.Parameters.cgroup: cgroup,
                        clickhouse_test_queues_processing.ClickhouseTestQueuesProcessing.Parameters.clickhouse_version: self.Parameters.test_version,
                        clickhouse_test_queues_processing.ClickhouseTestQueuesProcessing.Parameters.ch_config_version: self.Parameters.config_test_version,
                        clickhouse_test_queues_processing.ClickhouseTestQueuesProcessing.Parameters.ticket: self.Context.issue_key,
                    }
                )
                for cgroup in ['mtgiga-test', 'mtmobgiga-test']
            ], subtasks_variable=self.Context.queues_tasks, run=False)

            params = {
                clickhouse_mdb_acceptance.ClickhouseMdbAcceptance.Parameters.ch_version: 0,
                clickhouse_mdb_acceptance.ClickhouseMdbAcceptance.Parameters.pre_check: False,
                clickhouse_mdb_acceptance.ClickhouseMdbAcceptance.Parameters.upgrade: False,
            }

            self.run_subtasks([
                (
                    clickhouse_mdb_acceptance.ClickhouseMdbAcceptance,
                    dict(
                        description='Проверка совместимости кластера {} для приемки КХ {}'.format(cluster, self.Parameters.test_version),
                        ch_cluster=cluster,
                        ch_database=database,
                        ch_table=table,
                        ch_column=column,
                        **params
                    )
                )
                for cluster, database, table, column in clickhouse_mdb_acceptance.ClickhouseMdbAcceptance.CLUSTERS
            ], subtasks_variable=self.Context.mdb_tasks, wait=False)
