# coding=utf-8
import os

import six

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.utils import base_metrika_task, parameters as metrika_parameters, resource_types
from sandbox.projects.metrika.utils.mixins import console


class ClickHouseB2BStatBoxTrafQueries(sdk2.Resource):
    name = "CLICKHOUSE_B2B_STATBOX_TRAF_QUERIES"
    version = sdk2.resource.Attributes.String("Версия пакета, использованная для генерации запросов")
    date = sdk2.resource.Attributes.String("Дата, использованная для генерации запросов")


class ClickHouseB2BStatBoxTrafQueriesResource(resource_types.DefaultValueResourceMixin, sdk2.parameters.Resource):
    resource_type = ClickHouseB2BStatBoxTrafQueries.name
    owner = "METRIKA"


@base_metrika_task.with_parents
class ClickHouseStatboxTrafQueriesPrepare(base_metrika_task.BaseMetrikaTask, console.BaseConsoleMixin):
    name = "CLICKHOUSE_STATBOX_TRAF_QUERIES_PREPARE"

    class Requirements(base_metrika_task.BaseMetrikaTask.Requirements):
        privileged = True

    class Context(sdk2.Task.Context):
        version = None

    class Parameters(metrika_utils.CommonParameters):
        container = metrika_parameters.LastPeasantContainerResource("Environment container resource", required=True)
        date = sdk2.parameters.String("Дата", required=True,
                                      description="YYYY-MM-DD")
        force_pkg_version = sdk2.parameters.Bool("Задать явно версию пакета", required=True, default=False)
        with force_pkg_version.value[True]:
            pkg_version = sdk2.parameters.String("Версия пакета", required=True)

        with sdk2.parameters.Output:
            queries_resource = sdk2.parameters.Resource("Запросы для тестов", required=True)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        with sdk2.helpers.ProgressMeter("Add Repository"):
            self._add_repo()
        with sdk2.helpers.ProgressMeter("Install Package"):
            self._install_package()
        with sdk2.helpers.ProgressMeter("Create Queries"):
            self._create_queries()
        with sdk2.helpers.ProgressMeter("Create Resource"):
            self._create_resource()

    def _queries_dir(self):
        return self.path("queries")

    def _add_repo(self):
        with sdk2.Path("/etc/apt/sources.list.d/statbox-common.list").open('w', encoding='UTF-8', errors='replace') as f:
            f.write(u"deb http://dist.yandex.ru/statbox-common/ stable/all/\n")
            if self.Parameters.force_pkg_version:
                f.write(u"deb http://dist.yandex.ru/statbox-common/ testing/all/\n")

    def _install_package(self):
        self._execute_shell_and_check(["apt-get", "-qq", "update"])
        self._execute_shell_and_check(["apt-get", "--yes", "install", "python-pip"])
        self._execute_shell_and_check([
            "apt-get", "--yes", "install",
            "python-statface-client=0.154.0",
            "python-statinfra",
            "statbox-traf-action{}".format("=" + self.Parameters.pkg_version if self.Parameters.force_pkg_version else "")
        ])
        self._execute_shell_and_check("pip install --index-url https://pypi.yandex-team.ru/simple/ setuptools svn numpy==1.12.1 Cython")
        self._execute_shell_and_check("pip install --index-url https://pypi.yandex-team.ru/simple/ --ignore-installed six")
        self._execute_shell_and_check("pip install --index-url https://pypi.yandex-team.ru/simple/ pandas==0.19")

        version = sdk2.helpers.subprocess.check_output(["dpkg-query", "--showformat", "'${Version}'", "--show", "statbox-traf-action"])
        self.set_info("Версия пакета statbox-traf-action: {}".format(version))
        self.Context.version = version

    def _create_queries(self):
        environment = os.environ.copy()
        environment["STATBOX_ENVIRONMENT_SPEC"] = "mrparser"
        script_path = self.path("generate-queries.py")
        script_path.write_text(six.text_type(metrika_utils.read_sources_file("generate-queries.py")))
        self._execute_shell_and_check(["python", script_path.as_posix(), self.Parameters.date, self._queries_dir().as_posix()], env=environment)

    def _create_resource(self):
        queries_resource = ClickHouseB2BStatBoxTrafQueries(self, "", self._queries_dir().as_posix(), date=self.Parameters.date, version=self.Context.version)
        sdk2.ResourceData(queries_resource).ready()
        self.Parameters.queries_resource = queries_resource
