# coding=utf-8
import logging

import requests
from sandbox import common, sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters, resource_types
from sandbox.projects.metrika.utils.base_metrika_task import BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.metrika.utils.parameters import hide


class ClickhouseDistributionImportBinary(BaseConsoleMixin, BaseMetrikaTask):
    """
    Задача скачивает бинарь clickhouse-server указанной версии и кладёт в ресурс в Sandbox
    """

    class Parameters(CommonParameters):
        repository = sdk2.parameters.String("Репозиторий ClickHouse", required=True, default="packages.clickhouse.com/tgz",
                                            description="Адрес, по которому расположен репозиторий с tgz-архивами. https://clickhouse.com/docs/en/getting-started/install/#from-tgz-archives")
        version = sdk2.parameters.String("Версия ClickHouse", required=True,
                                         description="Версия, которую нужно импортировать, вида xx.yy.zz.gg")

        with sdk2.parameters.Output:
            binary_resource = sdk2.parameters.Resource('Бинарь ClickHouse', required=True,
                                                       description='Ресурс, содержащий импортированный бинарь')

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_execute(self):
        r = resource_types.MetrikaClickhouseBinary.find(attrs=dict(version=self.Parameters.version), state='READY').first()
        if r is None:
            self.perform_import()
        else:
            logging.info("Version %s already imported as resource %s.", self.Parameters.version, r.id)
            self.Parameters.binary_resource = r
            self.comment('<a href="{}">Бинарь ClickHouse</a>'.format(common.urls.get_resource_link(r.id)))

    def perform_import(self):
        package_name = 'clickhouse-common-static-{}'.format(self.Parameters.version)
        package_file = '{}-amd64.tgz'.format(package_name)
        for branch in ['stable', 'lts', 'prestable']:
            url = 'https://{repository}/{branch}/{package}'.format(repository=self.Parameters.repository, branch=branch, package=package_file)
            logging.debug("Trying url: %s", url)
            r = requests.head(url)
            if r.ok:
                logging.info("Found url: %s", url)
                break
        else:
            raise Exception('Clickhouse release {} not found in {}'.format(self.Parameters.version, self.Parameters.repository))

        self._execute_script('wget --no-verbose {} -O {}'.format(url, package_file))
        self._execute_script('tar -xzvf {}'.format(package_file))

        binary_archive = self.path('clickhouse.tar.gz')
        self._execute_script('tar -czvf {} -C {}/usr/bin clickhouse'.format(binary_archive.name, package_name))
        binary_resource = resource_types.MetrikaClickhouseBinary(
            self, 'clickhouse binary {}'.format(self.Parameters.version), binary_archive,
            version=self.Parameters.version, ttl='inf'
        )
        sdk2.ResourceData(binary_resource).ready()

        self.Parameters.binary_resource = binary_resource

        self.comment('<a href="{}">Бинарь ClickHouse</a>'.format(common.urls.get_resource_link(binary_resource.id)))
