# coding=utf-8

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.admins.metrika_packages_upload import MetrikaPackagesUpload
from sandbox.projects.metrika.utils import CommonParameters, parameters, settings
from sandbox.projects.metrika.utils.base_metrika_task import BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.metrika.utils.parameters import hide
from sandbox.sdk2.path import Path
from sandbox.common.types import resource as ctr

SOURCES_LIST = u"""
deb https://packages.clickhouse.com/deb stable main
deb https://packages.clickhouse.com/deb lts main
deb https://packages.clickhouse.com/deb prestable main
"""

PACKAGES = [
    "clickhouse-common-static",
    "clickhouse-server",
    "clickhouse-common-static-dbg",
    "clickhouse-client"
]

REPOS = [
    "yandex-trusty",
    "yandex-precise",
    "yandex-xenial"
]


class MetrikaClickhouseDebs(sdk2.Resource):
    any_arch = True
    auto_backup = True
    version = sdk2.Attributes.String('Version', required=True)
    restart_policy = ctr.RestartPolicy.IGNORE


class ClickhouseDistributionImportDeb(BaseConsoleMixin, BaseMetrikaTask):
    """
    Задача импортирует deb-пакеты указанной версии в указанный репозиторий
    """

    class Parameters(CommonParameters):
        container = parameters.LastPeasantContainerResource("Environment container resource", required=True)
        version = sdk2.parameters.String("Версия ClickHouse", required=True,
                                         description="Версия, которую нужно импортировать, вида xx.yy.zz.gg")

        packages = sdk2.parameters.List("Пакеты для скачивания", required=True, default=PACKAGES)

        destinations = sdk2.parameters.List("Репозитории назначения", required=True, default=REPOS)

        with sdk2.parameters.Output:
            packages_resource = sdk2.parameters.Resource('Deb-пакеты ClickHouse', required=True,
                                                         description='Ресурс, содержащий импортированные пакеты')

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    class Requirements(BaseMetrikaTask.Requirements):
        privileged = True

    @property
    def packages_dir(self):
        return self.path('packages')

    def on_execute(self):
        import retry
        import six
        import metrika.admin.python.scripts.deb_changes_generator.lib as lib
        import metrika.admin.python.scripts.deb_changes_generator.lib.view as view

        # 1. настроить репозиторий
        # 2. apt-get update
        # 3. apt-get download
        # 4. формируем changelog
        # 5. формируем ресурс
        # 6. запускаем таску загрузки и ждём

        with self.memoize_stage.download(commit_on_entrance=False):
            self._execute_shell_and_check("sudo apt-key adv --keyserver hkp://keyserver.ubuntu.com:80 --recv 8919F6BD2B48D754")

            sources_path = Path("/etc/apt/sources.list.d/clickhouse.list")
            sources_path.write_text(SOURCES_LIST)

            self.packages_dir.mkdir(parents=True)

            retry.retry_call(self.download_debs, tries=20)

            deb_infos = lib.examine_debs(self.packages_dir)
            changes_path = self.packages_dir / "{}_{}.changes".format("clickhouse", self.Parameters.version)
            changes_path.write_text(six.ensure_text(lib.render_changes(view.ViewModel(deb_infos, settings.username, source="clickhouse"))))

            packages_resource = MetrikaClickhouseDebs(self, "ClickHouse deb distribution {}".format(self.Parameters.version), self.packages_dir,
                                                      version=self.Parameters.version, ttl='inf')

            sdk2.ResourceData(packages_resource).ready()

            self.Parameters.packages_resource = packages_resource

        # далее - загрузка в репозиторий - там внутри свой memoize
        self.run_subtasks(
            [
                (MetrikaPackagesUpload,
                 {
                     MetrikaPackagesUpload.Parameters.artifacts.name: self.Parameters.packages_resource,
                     MetrikaPackagesUpload.Parameters.repository.name: repo,
                     MetrikaPackagesUpload.Parameters.wait_for_dist.name: True
                 }
                 ) for repo in self.Parameters.destinations
            ]
        )

    def download_debs(self):
        self._execute_shell_and_check("sudo apt-get --quiet update")
        self._execute_shell_and_check(["sudo", "apt-get", "download", "--quiet"] + ["{}={}".format(p, self.Parameters.version) for p in self.Parameters.packages], cwd=self.packages_dir.as_posix())
