# coding=utf-8

import logging
import time

from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.sdk2 import parameters


class MetrikaMdbChUpgradeParameters(CommonParameters):
    description = "Обновление версии ClickHouse в MDB для кластера"

    mdb_folder = parameters.String("MDB folder id", required=True, default="foori5uktoh2v12cbltq")
    ch_cluster = parameters.String("Имя кластера", required=True)
    ch_version = parameters.String("Версия ClickHouse на которую обновить", required=True)

    with parameters.Group("Секреты robot-metrika-admin") as admin_secrets_group:
        robot_admin_tokens_secret = parameters.YavSecret("Секрет с токенами robot-metrika-admin", required=True, default="sec-01cq6h07rwpqmqzb15y08jbs5q")

        robot_admin_token_key = parameters.String("Ключ OAuth токена в секрете", required=True, default="ymdb_oauth_token")

    _binary = binary_task.binary_release_parameters_list(stable=True)


@with_parents
class MetrikaMdbChUpgrade(BaseMetrikaTask):
    """
    Обновление версии ClickHouse в MDB для кластера
    """

    class Parameters(MetrikaMdbChUpgradeParameters):
        pass

    def on_execute(self):
        from metrika.pylib.yc.clickhouse import ManagedClickhouse

        token = self.Parameters.robot_admin_tokens_secret.data().get(self.Parameters.robot_admin_token_key)
        self.cluster = ManagedClickhouse(token=token).cluster_by_name(self.Parameters.ch_cluster, self.Parameters.mdb_folder)

        if self.cluster.data['config']['version'] == self.Parameters.ch_version:
            return

        self.check_updatable()

        update_mask = "configSpec.version"
        config_spec = self.cluster.data['config']
        config_spec['version'] = self.Parameters.ch_version
        try:
            operation = self.cluster.update(config_spec, update_mask)
        except Exception:
            logging.exception("Ошибка в процессе обновления конфигурации кластера.")
            raise TaskFailure(
                "Ошибка в процессе создания запроса на обновление конфигурации кластера, "
                "обратите внимание на логи."
            )
        self.wait_operation(operation)

    def check_updatable(self):
        target_version = self.Parameters.ch_version
        available_versions = self.cluster.available_versions()
        current_version = filter(lambda x: x['id'] == self.cluster.data['config']['version'], available_versions)
        if not current_version:
            raise TaskFailure(
                "Текущая версия {} не найдена в доступных версиях {}, "
                "обратитесь к администраторам MDB".format(self.cluster.data['config']['version'], available_versions)
            )
        current_version = current_version[0]

        if 'updatableTo' not in current_version:
            raise TaskFailure(
                "Невозможно определить список версий, на которые можно обновиться, "
                "обратитесь к администраторам MDB"
            )
        if target_version not in current_version['updatableTo']:
            raise TaskFailure(
                "Невозможно обновить текущую версию кластера до {}, "
                "список поддерживаемых версий для обновления: {}".format(target_version, current_version['updatableTo'])
            )

    def wait_operation(self, operation):
        done = False
        iteration = 0
        limit = 60
        operations_url = "https://yc.yandex-team.ru/folders/{}/managed-clickhouse/cluster/{}?section=operations".format(self.cluster.data['folderId'], self.cluster.data['id'])
        while not done:
            iteration += 1
            try:
                done = operation.get().get('done', False)
            except Exception:
                logging.exception("Ошибка в процессе получения статуса операции.")
            if iteration >= limit:
                raise TaskFailure(
                    "Операция {} не успела завершиться в отведенное время, "
                    "проверьте список операций по адресу {}".format(operation.data['id'], operations_url)
                )
            time.sleep(60)
