# coding=utf-8
import requests

from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.projects.common import constants
from sandbox.projects.common.build.YaPackage2 import YaPackage2Parameters
from sandbox.projects.common.build.ya_package_config.consts import PackageType
from sandbox.projects.metrika.utils.task_types.ya_package import MetrikaYaPackage
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.auth import MdbAuth, OAuth

DOCKER_API_URL = "https://registry.yandex.net/v2/"


def create_docker_registry_session():
    s = requests.session()
    s.auth = OAuth(sdk2.yav.Secret(settings.yav_uuid).data()['docker-token'])
    s.mount(DOCKER_API_URL, requests.adapters.HTTPAdapter(max_retries=Retry(total=20, backoff_factor=1)))
    return s


def is_docker_image_built(name, version):
    resp = create_docker_registry_session().get(DOCKER_API_URL + name + "/tags/list")
    resp.raise_for_status()
    return version in (resp.json().get("tags") or [])


def create_caas_session(user, password):
    s = requests.session()
    s.auth = MdbAuth(user, password)
    s.mount("http", requests.adapters.HTTPAdapter(max_retries=Retry(total=20, method_whitelist=frozenset(['POST']), status_forcelist=(500, 502, 504), backoff_factor=1)))
    s.verify = False
    return s


def build_clickhouse_docker_images(task, versions, without_merges=False):
    """
    :param sdk2.Task task:
    :param List[Tuple[str, str]] versions: Список троек (версия КХ, версия конфига КХ, версия YT словарей)
    :param bool without_merges: Нужно ли собирать версия с отключенными мержами

    Нужно собрать три докер-образа (или проверить, что они существуют)
    1. для caas вообще - т.к. там есть назначение портов - metrika/admin/maas/misc/docker/clickhouse/Dockerfile
        registry.yandex.net/clickhouse/caas:$version

    2. для caas в частности - словари yt - metrika/qa/docker/caas/Dockerfile
        registry.yandex.net/clickhouse/caas-metriqa:${ch-version}-yt-dict-${yt-dict-version}

    without_merges=True
    3. для caas в частности - словари yt и отключенные мержи - metrika/qa/docker/caas-no-merge/Dockerfile
        registry.yandex.net/clickhouse/caas-metriqa:${ch-version}-yt-dict-${yt-dict-version}-no-merge
    :return:
    """

    if not isinstance(versions, list):
        versions = [versions]

    common_params = {
        YaPackage2Parameters.checkout_arcadia_from_url: "arcadia:/arc/trunk/arcadia",
        YaPackage2Parameters.build_type: constants.PROFILE_BUILD_TYPE,
        YaPackage2Parameters.package_type: PackageType.DOCKER.value,
        YaPackage2Parameters.docker_push_image: True,
        YaPackage2Parameters.docker_user: settings.login,
        YaPackage2Parameters.docker_token_vault_name: settings.docker_registry_token,
    }

    task.Context.caas_versions = []
    if not task.Context.caas_subtasks:
        task.Context.caas_subtasks = []
    caas_subtasks = []

    for clickhouse_version, _, _ in versions:
        task.Context.caas_versions.append(clickhouse_version)
        if not is_docker_image_built(name='clickhouse/caas', version=task.Context.caas_versions[-1]):
            params = {
                "description": 'Сборка Docker-образа clickhouse-server для CaaS версии {}'.format(clickhouse_version),
                YaPackage2Parameters.packages: 'metrika/admin/maas/misc/docker/clickhouse/docker.json',
                YaPackage2Parameters.docker_image_repository: "clickhouse",
                YaPackage2Parameters.custom_version: task.Context.caas_versions[-1],
                YaPackage2Parameters.docker_build_arg: {"version": clickhouse_version}
            }
            caas_subtasks.append((MetrikaYaPackage, dict(params, **common_params)))

    task.run_subtasks(caas_subtasks, subtasks_variable=task.Context.caas_subtasks)

    task.Context.caas_yt_versions = []
    if not task.Context.caas_yt_subtasks:
        task.Context.caas_yt_subtasks = []
    caas_yt_subtasks = []

    for (clickhouse_version, ch_config_version, yt_dicts_version), caas_version in zip(versions, task.Context.caas_versions):
        task.Context.caas_yt_versions.append('{}-config-{}-yt-dict-{}'.format(caas_version, ch_config_version, yt_dicts_version))
        if not is_docker_image_built(name='clickhouse/caas-metriqa', version=task.Context.caas_yt_versions[-1]):
            params = {
                "description":
                    'Сборка Docker-образа clickhouse-server для CaaS версии {} c конфигом версии {} и словарями YT версии {}'.format(clickhouse_version, ch_config_version, yt_dicts_version),
                YaPackage2Parameters.packages: 'metrika/qa/docker/caas/docker.json',
                YaPackage2Parameters.docker_image_repository: "clickhouse",
                YaPackage2Parameters.custom_version: task.Context.caas_yt_versions[-1],
                YaPackage2Parameters.docker_build_arg:
                    {
                        "version": clickhouse_version,
                        "yt_dict_version": yt_dicts_version,
                        "ch_config_version": ch_config_version
                    }
            }
            caas_yt_subtasks.append((MetrikaYaPackage, dict(params, **common_params)))

    task.run_subtasks(caas_yt_subtasks, subtasks_variable=task.Context.caas_yt_subtasks)

    if without_merges:
        task.Context.caas_yt_no_merges_versions = []
        if not task.Context.caas_yt_no_merges_subtasks:
            task.Context.caas_yt_no_merges_subtasks = []
        caas_yt_no_merges_subtasks = []

        for (clickhouse_version, ch_config_version, yt_dicts_version), caas_yt_version in zip(versions, task.Context.caas_yt_versions):
            task.Context.caas_yt_no_merges_versions.append(caas_yt_version + '-no-merge')
            if not is_docker_image_built(name='clickhouse/caas-metriqa', version=task.Context.caas_yt_no_merges_versions[-1]):
                params = {
                    "description":
                        'Сборка Docker-образа clickhouse-server для CaaS версии {} c конфигом версии {}, словарями YT версии {} и отключенными мержами'.format(
                            clickhouse_version, ch_config_version, yt_dicts_version),
                    YaPackage2Parameters.packages: 'metrika/qa/docker/caas-no-merge/docker.json',
                    YaPackage2Parameters.docker_image_repository: "clickhouse",
                    YaPackage2Parameters.custom_version: task.Context.caas_yt_no_merges_versions[-1],
                    YaPackage2Parameters.docker_build_arg: {'version': caas_yt_version}
                }
                caas_yt_no_merges_subtasks.append((MetrikaYaPackage, dict(params, **common_params)))

        task.run_subtasks(caas_yt_no_merges_subtasks, subtasks_variable=task.Context.caas_yt_no_merges_subtasks)
