# coding=utf-8
import cgi
import distutils.dir_util
import itertools
import logging

from sandbox.projects.metrika.utils.mixins import maven
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import vcs
from sandbox.projects.metrika.utils.mixins import subtasks
from sandbox.projects.metrika.utils import parameters
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common import fs
from sandbox.common.types.task import Status, Semaphores
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.admins.cosmos.cosmos_launcher.view import ViewModel
from sandbox.projects.metrika.admins.cosmos.utils import CosmosArtifacts, TestFailures, HazelcastParameters, ReportStartrekParameters, YavParameters, CheckTestsResultParameters, PackInfo,\
    ReportTtlParameters
from sandbox.projects.metrika.admins.cosmos.cosmos_runner import CosmosRunner
from sandbox.projects.metrika.admins.cosmos.utils.packs import PacksDir
from sandbox.projects.metrika.utils import CommonParameters, render, custom_report_logger
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.sdk2 import Path

LOCAL_VERSION = '1.0-LOCAL'


@with_parents
class CosmosLauncher(BaseMetrikaTask, maven.BaseMavenMixin):
    """
    Запускатель паков
    """

    class Parameters(CommonParameters):
        container = parameters.LastCosmosContainerResource("Environment container resource", required=True)
        vcs_block = parameters.VcsParameters()
        with sdk2.parameters.Group("Сборка тестов") as build:
            build_configuration = sdk2.parameters.JSON('Конфигурация сборки', default={'pom.xml': {}}, required=True,
                                                       description='Конфигурация сборки артефактов. Каждый ключ - относительный путь к файлу pom.xml, значение - словарь Java-properties для сборки. '
                                                                   'Сборка осуществляется выполненением цели install для каждого указанного здесь файла.')
        with sdk2.parameters.Group("Прогон тестов") as run:
            packs_dirs = sdk2.parameters.List('Каталоги паков', default=['packs'], required=True,
                                              description='Относительный путь к каталогу с паками')
            run_configuration = sdk2.parameters.JSON("Конфигурация прогона", default={}, required=True,
                                                     description='Конфигурация прогона отдельных паков, каждый ключ - это имя пака, значение - словарь Java-properties для запуска. '
                                                                 'Запущены будут только паки, перечисленные здесь.')
            run_timeout = sdk2.parameters.Integer("Таймаут на прогон тестов, с", default=3 * 60 * 60, required=True,
                                                  description='Таймаут, который устанавливается на каждую дочернюю задачу.')
        report_ttl_params = ReportTtlParameters()
        yav = YavParameters()
        reporting = ReportStartrekParameters()
        hz = HazelcastParameters()
        check_test_results = CheckTestsResultParameters()
        semaphore_name = sdk2.parameters.String('Имя семафора')

        with sdk2.parameters.Output:
            with sdk2.parameters.Group("Результаты тестов") as output:
                test_results = sdk2.parameters.JSON("Результаты тестов", required=True,
                                                    description="Результаты тестов по всем пакам. Ключ - имя пака, значение - результаты прогона.")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(sdk2.Task.Context):
        artifact_resource_id = None
        packs = None
        runners = []

    def _bundle_dir(self, *path):
        return self.path('bundle', *path)

    def on_save(self):
        if self.Parameters.semaphore_name:
            self.Requirements.semaphores = Semaphores(
                acquires=[Semaphores.Acquire(self.Parameters.semaphore_name)],
                release=(Status.Group.BREAK, Status.Group.FINISH)
            )

    def on_execute(self):
        with sdk2.helpers.ProgressMeter("Build artifacts"):
            with self.memoize_stage.build:
                with sdk2.helpers.ProgressMeter("Checkout"):
                    vcs.checkout(self.Parameters.vcs_block, self.wd())
                self.Context.artifact_resource_id = self._build_artifacts()
        with sdk2.helpers.ProgressMeter("Run tests"):
            with self.memoize_stage.launch:
                self._run_tests()
            with self.memoize_stage.check_results:
                self._check_results()
        if self.Parameters.report_startrek:
            with sdk2.helpers.ProgressMeter("Report to Startrek issue {}".format(self.Parameters.issue_key)):
                self._report_startrek()

    def _build_artifacts(self):
        for pom_path, props in self.Parameters.build_configuration.iteritems():
            self._execute_maven(['versions:set', '--file', self.wd(pom_path)],
                                properties={'newVersion': LOCAL_VERSION}, cwd=self.wd())
            self._execute_maven(['install', '--threads', '1C', '--file', self.wd(pom_path)],
                                props, cwd=self.wd())

        fs.make_folder(self._bundle_dir())
        for packs_dir in self.Parameters.packs_dirs:
            self._prepare_packs(Path(self.wd(packs_dir)))
            distutils.dir_util.copy_tree(self.wd(packs_dir), self._bundle_dir('packs').as_posix())

        fs.copy_dir(self.path('.repository').as_posix(), self._bundle_dir('repository').as_posix())

        resource = CosmosArtifacts(self, '', self._bundle_dir().as_posix(), artifact_version=LOCAL_VERSION, ttl=7)
        sdk2.ResourceData(resource).ready()
        return resource.id

    def _prepare_packs(self, packs_dir):
        """
        :param packs_dir :type sdk2.Path
        """

        packs = PacksDir(packs_dir)
        for pack in packs.packs:
            if pack.name in self.Parameters.run_configuration:
                for artifact in pack.artifacts:
                    if not artifact.run_pom_path.exists():
                        with artifact.run_pom_path.open("w") as run_pom_file:
                            run_pom_file.write(render("run.pom.xml.jinja2", {"artifact": artifact.maven_version}))
                    self._execute_maven(['org.apache.maven.plugins:maven-dependency-plugin:3.1.1:go-offline', '--file',
                                         artifact.run_pom_path.as_posix()],
                                        cwd=self.wd())

                if not pack.report_pom_path.exists():
                    with pack.report_pom_path.open("w") as report_pom_file:
                        report_pom_file.write(render("report.pom.xml.jinja2"))
                self._execute_maven(['org.apache.maven.plugins:maven-dependency-plugin:3.1.1:go-offline', '--file',
                                     pack.report_pom_path.as_posix()],
                                    cwd=self.wd())

    def after_subtasks_enqueued(self):
        for (pack_id, pack_state), task_id in zip(sorted(self.Context.packs.iteritems()), self.Context.runners):
            PackInfo(pack_state).task_id = task_id

    def _run_tests(self):
        # парсим каталог с паками
        # запускаем дочерние задачи
        self.Context.packs = {
            pack_dir.name: PackInfo().init(pack_dir).state
            for pack_dir in
            list(itertools.chain(*[Path(self.wd(packs_dir)).iterdir() for packs_dir in self.Parameters.packs_dirs]))
            if pack_dir.name in self.Parameters.run_configuration
        }

        if not self.Context.packs:
            self.set_info("Ни одного пака с тестами не будет запущено. Возможно ошибка в параметрах.")

        params = {
            CosmosRunner.Parameters.artifacts_resource.name: self.Context.artifact_resource_id,
            CosmosRunner.Parameters.kill_timeout.name: self.Parameters.run_timeout
        }
        params.update(self.Parameters.report_ttl_params)
        params.update(self.Parameters.yav)
        params.update(self.Parameters.hz)

        tasks = []
        for pack_id, pack_state in sorted(self.Context.packs.iteritems()):
            task_params = params.copy()
            task_params.update({
                CosmosRunner.Parameters.pack_id.name: pack_id,
                CosmosRunner.Parameters.description.name: PackInfo(pack_state).title,
                CosmosRunner.Parameters.test_properties.name: self.Parameters.run_configuration[pack_id]
            })
            tasks.append((CosmosRunner, task_params))
        self.run_subtasks(tasks, subtasks_variable=self.Context.runners)

    def _check_results(self):
        if self.Context.packs:
            tasks = [self.find(id=PackInfo(pack_state).task_id).first() for pack_state in self.Context.packs.values()]
            failed = [task for task in tasks if task.status not in Status.Group.SUCCEED]
            if failed:
                raise subtasks.SubtasksException(failed)

            # забрать результаты из дочерних задач
            self.Parameters.test_results = {
                name: {
                    "title": PackInfo(pack_state).title,
                    "result": self.find(id=PackInfo(pack_state).task_id).first().Parameters.test_results
                } for name, pack_state in self.Context.packs.iteritems()
            }

            if self.Parameters.fail_task_on_test_failure:
                if not self.is_all_tests_passed:
                    raise TestFailures()

    def _report_startrek(self):
        # этот код не должен ничего ломать
        try:
            issue = self.st_client.issues[self.Parameters.issue_key]
            issue.comments.create(text=self.get_wiki_comment())
        except Exception as e:
            self.set_info(e.message)
            logging.warning("Ошибка в отправке комментария в Startrek", exc_info=True)

    @sdk2.header()
    @custom_report_logger
    def header(self):
        return render("header.html.jinja2", {"view": ViewModel(self)})

    @sdk2.report(utils.display("wiki-комментарий"))
    @custom_report_logger
    def wiki_comment(self):
        return "<pre>" + self.get_wiki_comment() + "</pre>"

    @sdk2.report(utils.display("html-комментарий"))
    @custom_report_logger
    def html_comment(self):
        return "<pre>" + cgi.escape(self.get_html_comment()) + "</pre>"

    def get_wiki_comment(self):
        return render("wiki.txt.jinja2", {"view": ViewModel(self)})

    def get_html_comment(self):
        return render("email.html.jinja2", {"view": ViewModel(self)})

    @property
    def is_all_tests_passed(self):
        if self.Parameters.test_results:
            return sum(int(tests["result"]["failed"]) for tests in self.Parameters.test_results.values()) == 0
        else:
            raise errors.TaskError("Результаты тестов не доступны")
