# coding=utf-8
import logging
import shutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common import fs
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.admins.cosmos.utils import CosmosReport, ReportNotGenerated, ReportTtlParameters
from sandbox.projects.metrika.admins.cosmos.utils import PackDir
from sandbox.projects.metrika.admins.cosmos.utils.surefire_report import parse_surefire_reports
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils import parameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins import maven
from sandbox.sdk2 import paths, Path


@with_parents
class CosmosReporter(BaseMetrikaTask, maven.BaseMavenMixin):
    """
    Генератор отчётов по прогонам тестов, используется в крайних случаях, когда нужно перегенерировать отчёт
    """

    class Parameters(CommonParameters):
        container = parameters.LastCosmosContainerResource("Environment container resource", required=True)
        pack_id = sdk2.parameters.String('Идентификатор пака', required=True,
                                         description='Имя каталога, содержащего описание пака для генерации отчёта')
        artifacts_resource = sdk2.parameters.Resource('Артефакты тестов', required=True,
                                                      description='Ресурс, содержащий локальный maven-репозиторий с необходимыми для данного пака артефактами и каталог с описаниями паков')
        old_reports_resource = sdk2.parameters.Resource('Отчёты', required=True,
                                                        description='Ресурс, содержащий отчёты после прогона пака')
        report_ttl_params = ReportTtlParameters()
        with sdk2.parameters.Output:
            with sdk2.parameters.Group('Результаты тестов') as output:
                new_report_resource = sdk2.parameters.Resource('Отчёты', required=True,
                                                               description='Ресурс, содержащий отчёты после повторной генерации отчёта')
                test_results = sdk2.parameters.JSON('Результаты тестов', required=True,
                                                    description='Краткие результаты тестов')

        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(sdk2.Task.Context):
        name = None
        title = None

    def on_execute(self):
        with sdk2.helpers.ProgressMeter("Download bundle"):
            self._download_bundle()
            self.Context.name = self._pack().name
            self.Context.title = self._pack().title
            self.Context.save()

        with sdk2.helpers.ProgressMeter("Download report"):
            self._download_report()

        with sdk2.helpers.ProgressMeter("ReBuild report"):
            self._re_build_report()

    def _new_report_path(self):
        return self.path('report')

    def _old_report_path(self):
        return self.path('old-report')

    def _pack(self):
        return PackDir(Path(self.wd('packs', self.Parameters.pack_id)))

    def _get_maven_local_repo_path(self):
        return Path(self.wd("repository"))

    def _download_bundle(self):
        fs.copy_dir(sdk2.ResourceData(self.Parameters.artifacts_resource).path.as_posix(), self.wd())
        paths.add_write_permissions_for_path(self.wd())

    def _download_report(self):
        fs.copy_dir(sdk2.ResourceData(self.Parameters.old_reports_resource).path.as_posix(), self._old_report_path().as_posix())
        paths.add_write_permissions_for_path(self._old_report_path().as_posix())

    def _re_build_report(self):
        shutil.move(self._old_report_path().as_posix(), self._pack().target().as_posix())
        try:
            # junit файлы распарсим и получим результат, тут возможен невалидный xml и т.п.
            self.Parameters.test_results = parse_surefire_reports(self._pack().target("surefire-reports"))

            # выполним генерацию репорта если тут упадёт, то сохраним таки артефакт, что бы потом что-то как-то перегенировать
            self._execute_maven(['site', '--file', self._pack().report_pom_path.as_posix()])
        except:
            logging.warning("Ошибка в генерации отчёта, чего быть не должно.", exc_info=True)
            raise ReportNotGenerated()
        finally:
            # заархивируем в артефакт то, что удалось сделать - пригодится для дальнейших разбирательств
            shutil.move(self._pack().target().as_posix(), self._new_report_path().as_posix())
            report_resource = CosmosReport(self, self._pack().title, self._new_report_path(), ttl=self.Parameters.report_ttl,
                                           artifact_resource_id=self.Parameters.artifacts_resource.id,
                                           pack_id=self._pack().name)

            sdk2.ResourceData(report_resource).ready()

            self.Parameters.new_report_resource = report_resource

    @property
    def is_all_tests_passed(self):
        if self.Parameters.test_results:
            return int(self.Parameters.test_results["failed"]) == 0
        else:
            raise errors.TaskError("Результаты тестов не доступны")
